import type React from "react"
import type { Metadata } from "next"
import { GeistSans } from "geist/font/sans"
import { GeistMono } from "geist/font/mono"
import { ThemeProvider } from "@/components/theme-provider"
import { SettingsProvider } from "@/components/settings-provider"
import { AuthProvider } from "@/components/auth-provider"
import { SettingsContextProvider } from "@/context/SettingsContext";
// import { QueryClient, QueryClientProvider } from "@tanstack/react-query";
// import { ReactNode, useState } from "react";

import "./globals.css"

// Static metadata - this will be the default/fallback
export const metadata: Metadata = {
  title: "IndzsSearch - Fast, Comprehensive Search Engine",
  description: "Search the web, images, and videos with IndzsSearch. Fast, reliable search results powered by advanced technology.",
  generator: "IndzsSearch",
  keywords: ["search engine", "web search", "image search", "video search", "IndzsSearch"],
  authors: [{ name: "IndzsSearch Team" }],
  creator: "IndzsSearch",
  publisher: "IndzsSearch",
  robots: "index, follow",
  openGraph: {
    type: "website",
    locale: "en_US",
    url: process.env.NEXT_PUBLIC_APP_URL || "https://indussearch.com",
    siteName: "IndzsSearch",
    title: "IndzsSearch - Fast, Comprehensive Search Engine",
    description: "Search the web, images, and videos with IndzsSearch. Fast, reliable search results powered by advanced technology.",
    images: [
      {
        url: "/logo.png",
        width: 120,
        height: 40,
        alt: "IndzsSearch Logo",
      },
    ],
  },
  twitter: {
    card: "summary_large_image",
    title: "IndzsSearch - Fast, Comprehensive Search Engine",
    description: "Search the web, images, and videos with IndzsSearch. Fast, reliable search results powered by advanced technology.",
    images: ["/logo.png"],
  },
  icons: {
    icon: "/favicon.png",
    shortcut: "/favicon.png",
    apple: "/favicon.png",
  },
}

export default function RootLayout({
  children,
}: Readonly<{
  children: React.ReactNode
}>) {
  // const [queryClient] = useState(() => new QueryClient());

  return (
    <html lang="en" suppressHydrationWarning>
      <head>
        <script
          dangerouslySetInnerHTML={{
            __html: `
              try {
                // Apply theme from cookie or system preference
                const cookies = document.cookie.split(';');
                const themeCookie = cookies.find(cookie => cookie.trim().startsWith('indus-search-theme='));
                let theme = 'system';
                
                if (themeCookie) {
                  theme = decodeURIComponent(themeCookie.split('=')[1] || 'system');
                }
                
                if (theme === 'dark' || (theme === 'system' && window.matchMedia && window.matchMedia('(prefers-color-scheme: dark)').matches)) {
                  document.documentElement.classList.add('dark');
                } else {
                  document.documentElement.classList.add('light');
                }

                // Fetch and apply dynamic site settings (client-side)
                fetch('/api/site-settings')
                  .then(response => {
                    if (!response.ok) throw new Error('Network response was not ok');
                    return response.json();
                  })
                  .then(settings => {
                    try {
                      // Update page title
                      if (settings && settings.siteTitle) {
                        document.title = settings.siteTitle;
                      }

                      // Update meta description
                      if (settings && settings.siteDescription) {
                        const metaDesc = document.querySelector('meta[name="description"]');
                        if (metaDesc) {
                          metaDesc.setAttribute('content', settings.siteDescription);
                        }
                      }

                      // Update meta keywords
                      if (settings && settings.siteKeywords) {
                        const metaKeywords = document.querySelector('meta[name="keywords"]');
                        if (metaKeywords) {
                          metaKeywords.setAttribute('content', settings.siteKeywords);
                        } else {
                          const meta = document.createElement('meta');
                          meta.name = 'keywords';
                          meta.content = settings.siteKeywords;
                          document.head.appendChild(meta);
                        }
                      }

                      // Update Open Graph tags
                      if (settings && settings.siteTitle) {
                        const ogTitle = document.querySelector('meta[property="og:title"]');
                        if (ogTitle) {
                          ogTitle.setAttribute('content', settings.siteTitle);
                        }
                      }

                      if (settings && settings.siteDescription) {
                        const ogDesc = document.querySelector('meta[property="og:description"]');
                        if (ogDesc) {
                          ogDesc.setAttribute('content', settings.siteDescription);
                        }
                      }

                      // Update theme color
                      if (settings && settings.themeColor) {
                        const existingMeta = document.querySelector('meta[name="theme-color"]');
                        if (existingMeta) {
                          existingMeta.setAttribute('content', settings.themeColor);
                        } else {
                          const meta = document.createElement('meta');
                          meta.name = 'theme-color';
                          meta.content = settings.themeColor;
                          document.head.appendChild(meta);
                        }
                      }

                      // Update favicon
                      if (settings && settings.favicon) {
                        const existingIcons = document.querySelectorAll('link[rel~="icon"]');
                        //existingIcons.forEach(el => el.parentNode?.removeChild(el));
                        const link = document.createElement('link');
                        link.rel = 'icon';
                        link.href = settings.favicon;
                        link.type = 'image/x-icon';
                        document.head.appendChild(link);
                      }
                    } catch(e) {
                      console.error('Failed to apply site settings to head:', e);
                    }
                  })
                  .catch(e => console.error('Failed to load site settings:', e));
              } catch (e) {
                // avoid blocking render on errors
                console.error('Theme script error:', e);
              }
            `,
          }}
        />
      </head>
      <body className={`font-sans ${GeistSans.variable} ${GeistMono.variable}`}>
        <ThemeProvider defaultTheme="system" storageKey="indus-search-theme">
          <SettingsContextProvider>
            {/* <QueryClientProvider client={queryClient}> */}
              <SettingsProvider>
                <AuthProvider>{children}</AuthProvider>
              </SettingsProvider>
            {/* </QueryClientProvider> */}
          </SettingsContextProvider>
        </ThemeProvider>
      </body>
    </html>
  )
}
