import { type NextRequest, NextResponse } from "next/server";

interface WikipediaSearchResult {
  query: {
    search: {
      title: string;
      snippet: string;
      pageid: number;
    }[];
  };
}

interface WikipediaSummary {
  extract?: string;
  title: string;
  description?: string;
  content_urls?: {
    desktop?: { page: string };
    mobile?: { page: string };
  };
}


import axios from "axios";
import * as cheerio from "cheerio";

async function scrapeWikipediaImagesByPageId(pageId) {
   // 2. Fetch page images
    const imagesRes = await fetch(
      `https://en.wikipedia.org/w/api.php?action=query&pageids=${pageId}&prop=images&format=json`
    );
    const imagesData = await imagesRes.json();

    const imageTitles = imagesData.query.pages[pageId].images
      ?.map((img: any) => img.title)
      .filter((title: string) => /\.(jpg|jpeg|png|gif)$/i.test(title));

    if (!imageTitles?.length) {
      return []
    }

    // 3. Get full URLs for the first few images
    const titlesParam = imageTitles.slice(0, 5).map((t: string) => encodeURIComponent(t)).join("|");
    const imageInfoRes = await fetch(
      `https://en.wikipedia.org/w/api.php?action=query&titles=${titlesParam}&prop=imageinfo&iiprop=url&format=json`
    );
    const imageInfoData = await imageInfoRes.json();

    const urls: string[] = Object.values(imageInfoData.query.pages)
      .map((page: any) => page.imageinfo?.[0]?.url)
      .filter(Boolean);
    return urls
}

// import fetch from "node-fetch";
// import cheerio from "cheerio";

// import axios from "axios";
// import cheerio from "cheerio";

// import axios from "axios";
// import cheerio from "cheerio";

// export async function scrapeWikipediaImagesByPageId(pageId: string | number) {
//   try {
//     const pageUrl = `https://en.wikipedia.org/?curid=${pageId}`;
//     const res = await axios.get(pageUrl);
//     const html = res.data;

//     const $ = cheerio.load(html);
//     const urls: string[] = [];

//     // 1️⃣ Get infobox images (main image)
//     $("table.infobox img").each((_i, img) => {
//       const src = $(img).attr("src");
//       const fileWidth = parseInt($(img).attr("data-file-width") || "0");
//       const fileHeight = parseInt($(img).attr("data-file-height") || "0");

//       if (src && fileWidth > 50 && fileHeight > 50) {
//         const fullUrl = src.startsWith("//") ? `https:${src}` : src;
//         if (!urls.includes(fullUrl)) urls.push(fullUrl);
//       }
//     });

//     // 2️⃣ Get other images in content area
//     $("#mw-content-text img").each((_i, img) => {
//       const src = $(img).attr("src");
//       const fileWidth = parseInt($(img).attr("data-file-width") || "0");
//       const fileHeight = parseInt($(img).attr("data-file-height") || "0");

//       if (src && fileWidth > 50 && fileHeight > 50) {
//         const fullUrl = src.startsWith("//") ? `https:${src}` : src;
//         if (!urls.includes(fullUrl)) urls.push(fullUrl);
//       }
//     });

//     // 3️⃣ Return first 5 images
//     return urls.slice(0, 5);
//   } catch (err) {
//     console.error("Error scraping Wikipedia images:", err);
//     return [];
//   }
// }



/**
 * GET handler for fetching a Wikipedia summary.
 * Example usage: /api/wikipedia/summary?q=tiger
 */
export async function POST(req: NextRequest) {
  // const query = req.nextUrl.searchParams.get("q");
  const { query } = await req.json();
  console.log(query)
  if (!query) {
    return NextResponse.json(
      { error: "Query parameter 'q' is required" },
      { status: 400 }
    );
  }

  try {
    // 1️⃣ Search Wikipedia for matching pages
    const searchRes = await fetch(
      `https://en.wikipedia.org/w/api.php?action=query&list=search&srsearch=${encodeURIComponent(
        query
      )}&format=json&origin=*`
    );

    if (!searchRes.ok) {
      console.error("Wikipedia search request failed:", searchRes.statusText);
      return NextResponse.json(
        { error: "Wikipedia search request failed" },
        { status: 500 }
      );
    }

    const searchData: WikipediaSearchResult = await searchRes.json();

    if (!searchData.query.search.length) {
      return NextResponse.json(
        { error: "No Wikipedia articles found" },
        { status: 404 }
      );
    }

    const title = searchData.query.search[0].title;
    console.log(title)
    console.log(searchData.query.search[0].pageid)

    // 2️⃣ Fetch summary for the top result
    const summaryRes = await fetch(
      `https://en.wikipedia.org/api/rest_v1/page/summary/${encodeURIComponent(
        title
      )}`
    );

    if (!summaryRes.ok) {
      console.error("Wikipedia summary request failed:", summaryRes.statusText);
      return NextResponse.json(
        { error: "Wikipedia summary request failed" },
        { status: 500 }
      );
    }

    const summaryData: WikipediaSummary = await summaryRes.json();

    if (!summaryData.extract) {
      return NextResponse.json(
        { error: "No summary available for this article" },
        { status: 404 }
      );
    }

    const images = await scrapeWikipediaImagesByPageId(searchData.query.search[0].pageid); // Example pageId
    console.log(images);

    // 3️⃣ Return summary
    return NextResponse.json({
      title: summaryData.title,
      summary: summaryData.extract,
      description: summaryData.description || null,
      pageUrl: summaryData.content_urls?.desktop?.page || null,
      images : images || null
    });
  } catch (error) {
    console.error("Error fetching Wikipedia summary:", error);
    return NextResponse.json(
      { error: "An unexpected error occurred" },
      { status: 500 }
    );
  }
}
