import { type NextRequest, NextResponse } from "next/server"
import { createUserAdsCollection } from "@/lib/typesense"
import { client } from "@/lib/typesense"

export async function GET(request: NextRequest) {
  try {
    const { searchParams } = new URL(request.url)
    const userId = searchParams.get("userId")

    if (!userId) {
      return NextResponse.json({ error: "User ID is required" }, { status: 400 })
    }

    await createUserAdsCollection()

    const searchResults = await client
      .collections("user_ads")
      .documents()
      .search({
        q: "*",
        filter_by: `userId:=${userId}`,
        per_page: 100,
      })

    const ads = searchResults.hits?.map((hit: any) => hit.document) || []

    return NextResponse.json({ ads })
  } catch (error) {
    console.error("Error fetching user ads:", error)
    return NextResponse.json({ error: "Internal server error" }, { status: 500 })
  }
}

export async function POST(request: NextRequest) {
  try {
    const adData = await request.json()

    if (!adData.userId || !adData.title || !adData.description || !adData.targetUrl) {
      return NextResponse.json({ error: "Missing required fields" }, { status: 400 })
    }

    await createUserAdsCollection()

    const newAd = {
      ...adData,
      id: Math.random().toString(36).substr(2, 9),
      totalSpent: 0,
      clicks: 0,
      impressions: 0,
      dailyShown: 0,
      monthlyShown: 0,
      lastShownDate: "",
      isActive: true,
      createdAt: new Date().toISOString(),
    }

    const result = await client.collections("user_ads").documents().create(newAd)

    return NextResponse.json({ success: true, ad: result })
  } catch (error) {
    console.error("Error creating user ad:", error)
    return NextResponse.json({ error: "Internal server error" }, { status: 500 })
  }
}
