import { type NextRequest, NextResponse } from "next/server"
import { submitWebsite } from "@/lib/typesense"

export async function POST(request: NextRequest) {
  try {
    const body = await request.json()

    const { title, url, imageUrl, supportImageCrawl, supportWebCrawl, ownerConfirmed } = body

    // Validation
    if (!title || !url || !ownerConfirmed) {
      return NextResponse.json({ error: "Missing required fields" }, { status: 400 })
    }

    // Basic URL validation
    try {
      new URL(url)
    } catch {
      return NextResponse.json({ error: "Invalid URL format" }, { status: 400 })
    }

    // Submit to Typesense
    const result = await submitWebsite({
      title: title.trim(),
      url: url.trim(),
      imageUrl: imageUrl?.trim() || undefined,
      supportImageCrawl: Boolean(supportImageCrawl),
      supportWebCrawl: Boolean(supportWebCrawl),
      ownerConfirmed: Boolean(ownerConfirmed),
    })

    console.log("[SUBMIT WEBSITE] Successfully submitted:", { title, url })

    return NextResponse.json({
      success: true,
      message: "Website submitted successfully",
      id: result.id,
    })
  } catch (error) {
    console.error("[SUBMIT WEBSITE] Error:", error)
    return NextResponse.json({ error: "Failed to submit website" }, { status: 500 })
  }
}
