import { type NextRequest, NextResponse } from "next/server"
import { searchWeb, searchImages, searchVideos, searchAllAds } from "@/lib/typesense"

export async function GET(request: NextRequest) {
  try {
    const { searchParams } = new URL(request.url)
    const query = searchParams.get("q") || ""
    const type = searchParams.get("type") || "web"
    const safeSearch = searchParams.get("safesearch") === "true"
    const dateFilter = searchParams.get("date") || "all"
    const page = Number.parseInt(searchParams.get("page") || "1")
    const limit = Number.parseInt(searchParams.get("limit") || "25")

    let results: any[] = []
    let total = 0
    let hasMore = false

    if (type === "web") {
      // Get web search results
      const webResults = await searchWeb({
        query,
        type: "web",
        safeSearch,
        dateFilter,
        page,
        limit,
      })

      // Get ads to mix into results (both admin and user ads)
      let adsToShow: any[] = []
      if (page === 1) { // Only show ads on first page
        try {
          const adsResult = await searchAllAds({
            query,
            type: "ads", 
            safeSearch,
            dateFilter: "all",
            limit: 4, // Limit ads per page
          })
          
          adsToShow = adsResult.results.map(ad => ({
            ...ad,
            isAd: true, // Mark as ad for the frontend
            // Preserve isUserAd property from searchAllAds
          }))
          
          console.log("[SEARCH API] Mixed ads:", {
            totalAds: adsToShow.length,
            adminAds: adsToShow.filter(ad => !ad.isUserAd).length,
            userAds: adsToShow.filter(ad => ad.isUserAd).length,
          })
        } catch (adError) {
          console.error("[SEARCH API] Error fetching ads:", adError)
        }
      }

      // Mix ads into web results (e.g., after positions 0, 3, 7, etc.)
      const mixedResults: any[] = []
      let adIndex = 0
      
      for (let i = 0; i < webResults.results.length; i++) {
        // Add web result
        mixedResults.push(webResults.results[i])
        
        // Insert ad after certain positions (0, 3, 7, 11, etc.)
        if (page === 1 && (i === 0 || (i + 1) % 4 === 0) && adIndex < adsToShow.length) {
          mixedResults.push(adsToShow[adIndex])
          adIndex++
        }
      }

      results = mixedResults
      total = webResults.total
      hasMore = webResults.hasMore

    } else if (type === "images") {
      const imageResults = await searchImages({
        query,
        type: "images",
        safeSearch,
        dateFilter,
        page,
        limit,
      })
      results = imageResults.results
      total = imageResults.total
      hasMore = imageResults.hasMore

    } else if (type === "videos") {
      const videoResults = await searchVideos({
        query,
        type: "videos", 
        safeSearch,
        dateFilter,
        page,
        limit,
      })
      results = videoResults.results
      total = videoResults.total
      hasMore = videoResults.hasMore
    }

    return NextResponse.json({
      results,
      total,
      hasMore,
      adsCount: results.filter(r => r.isAd).length,
    })

  } catch (error) {
    console.error("[SEARCH API] Error:", error)
    return NextResponse.json(
      {
        results: [],
        total: 0,
        hasMore: false,
        error: error instanceof Error ? error.message : "Search failed",
      },
      { status: 500 }
    )
  }
}
