import { type NextRequest, NextResponse } from "next/server"

export async function POST(request: NextRequest) {
  try {
    const { amount, currency = "INR", userId } = await request.json()

    if (!amount || amount < 100) {
      return NextResponse.json({ error: "Minimum amount is ₹1" }, { status: 400 })
    }

    if (!userId) {
      return NextResponse.json({ error: "User ID is required" }, { status: 400 })
    }

    const razorpayKeyId = process.env.RAZORPAY_KEY_ID
    const razorpayKeySecret = process.env.RAZORPAY_KEY_SECRET

    if (!razorpayKeyId || !razorpayKeySecret) {
      console.error("Razorpay credentials not configured")
      return NextResponse.json({ error: "Payment service not configured" }, { status: 500 })
    }

    const orderData = {
      amount, // Razorpay expects amount in paise
      currency,
      receipt: `receipt_${userId}_${Date.now()}`,
      notes: {
        userId: userId,
        purpose: "wallet_recharge",
      },
    }

    const response = await fetch("https://api.razorpay.com/v1/orders", {
      method: "POST",
      headers: {
        "Content-Type": "application/json",
        Authorization: `Basic ${Buffer.from(`${razorpayKeyId}:${razorpayKeySecret}`).toString("base64")}`,
      },
      body: JSON.stringify(orderData),
    })

    if (!response.ok) {
      const errorData = await response.json()
      console.error("Razorpay API error:", errorData)
      throw new Error(`Razorpay API error: ${response.status}`)
    }

    const order = await response.json()

    return NextResponse.json({
      id: order.id,
      amount: order.amount, // Keep amount in paise for Razorpay frontend
      currency: order.currency,
      status: order.status,
      razorpayKeyId,
    })
  } catch (error) {
    console.error("Error creating Razorpay order:", error)
    return NextResponse.json(
      {
        error: "Failed to create payment order. Please try again.",
      },
      { status: 500 },
    )
  }
}
