import { type NextRequest, NextResponse } from "next/server"
import { client } from "@/lib/typesense"

export async function GET(request: NextRequest) {
  try {
    const { searchParams } = new URL(request.url)
    const query = searchParams.get("q")
    console.log(request.signal)

    if (!query || query.trim().length < 1) {
      return NextResponse.json({ suggestions: [] })
    }

    const searchResults = await client.collections("web_results").documents().search({
      q: query.trim(),
      query_by: "keywords,title,description", // Search across all three fields
      limit: 12,
      prefix: true, // Enable prefix matching for faster autocomplete
      drop_tokens_threshold: 0, // Don't drop tokens for better matching
      typo_tokens_threshold: 0, // Disable typo tolerance for speed
      num_typos: 0, // No typos for faster results
      sort_by: "_text_match:desc",
    })

    const suggestions = new Set<string>()

    searchResults.hits?.forEach((hit: any) => {
      // Add keywords first (higher priority)
      if (hit.document.keywords && Array.isArray(hit.document.keywords)) {
        hit.document.keywords.forEach((keyword: string) => {
          if (keyword.toLowerCase().startsWith(query.toLowerCase())) {
            suggestions.add(keyword)
          }
        })
      }

      // Add titles that start with query
      const title = hit.document.title
      if (title && title.toLowerCase().startsWith(query.toLowerCase())) {
        suggestions.add(title)
      }

      // Add relevant phrases from description
      const description = hit.document.description
      if (description) {
        const words = description.toLowerCase().split(/\s+/)
        const queryLower = query.toLowerCase()

        // Find words that start with the query
        words.forEach((word) => {
          if (word.startsWith(queryLower) && word.length > queryLower.length) {
            suggestions.add(word)
          }
        })

        // Also check if description starts with query
        if (description.toLowerCase().startsWith(queryLower)) {
          // Extract first few words as suggestion
          const firstPhrase = description.split(" ").slice(0, 4).join(" ")
          if (firstPhrase.length > query.length) {
            suggestions.add(firstPhrase)
          }
        }
      }
    })



    // =================================
    try {
      const res = await fetch(
        `https://suggestqueries.google.com/complete/search?client=firefox&q=${encodeURIComponent(
          query
        )}`,
        {
          headers: {
            'User-Agent': 'Mozilla/5.0',
          },
          signal: request.signal, // ← THIS enables abort
        }
      )

      const data = await res.json()
      const googleSuggestions = Array.isArray(data?.[1]) ? data[1] : []


      // const finalSuggestions = Array.from(suggestions).slice(0, 8)
      const finalSuggestions = Array.from(
        new Set([...suggestions, ...googleSuggestions])
      ).slice(0, 8)

      return NextResponse.json({
        suggestions: finalSuggestions,
        query: query.trim(),
      })
    } catch (e) {
      const finalSuggestions = Array.from(suggestions).slice(0, 8)
      return NextResponse.json({
        suggestions: finalSuggestions,
        query: query.trim(),
      })

    }
  } catch (error) {
    console.error("Autocomplete error:", error)
    return NextResponse.json({ suggestions: [], query: "" })
  }
}
