import { type NextRequest, NextResponse } from "next/server"
import { createUser, checkUsernameAvailability } from "@/lib/typesense"
import { sendVerificationEmail } from "@/lib/email"

export async function POST(request: NextRequest) {
  try {
    const { username, email, password } = await request.json()

    if (!username || !email || !password) {
      return NextResponse.json({ error: "All fields are required" }, { status: 400 })
    }

    if (password.length < 6) {
      return NextResponse.json({ error: "Password must be at least 6 characters long" }, { status: 400 })
    }

    // Validate email format
    const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/
    if (!emailRegex.test(email)) {
      return NextResponse.json({ error: "Please enter a valid email address" }, { status: 400 })
    }

    // Check username availability
    const isAvailable = await checkUsernameAvailability(username)
    if (!isAvailable) {
      return NextResponse.json({ error: "Username is already taken" }, { status: 400 })
    }

    // Create user
    const result = await createUser({ username, email, password })

    if (!result.success) {
      return new Response(
        JSON.stringify({
          success: false,
          message: result.error // This will be "Email already exists"
        }),
        { status: 400, headers: { "Content-Type": "application/json" } }
      );
    }

    const user = result.data
    console.log(user.emailVerificationToken)
    // Send verification email
    try {
      if (user.emailVerificationToken) {
        console.log(user.emailVerificationToken)
        await sendVerificationEmail(email, username, user.emailVerificationToken)
        console.log(`[v0] Verification email sent to ${email}`)
      }
    } catch (emailError) {
      console.error("Failed to send verification email:", emailError)
      // Continue with registration even if email fails
    }

    // Remove password and sensitive fields from response
    const { password: _, emailVerificationToken: __, ...userWithoutPassword } = user

    return NextResponse.json({
      success: true,
      message: "Registration successful! Please check your email to verify your account.",
      user: userWithoutPassword,
      requiresVerification: true,
    })
  } catch (error) {
    console.error("Registration error:", error)
    return NextResponse.json({ error: "Internal server error" }, { status: 500 })
  }
}
