import { type NextRequest, NextResponse } from "next/server"
import { authenticateUser } from "@/lib/typesense"

export async function POST(request: NextRequest) {
  try {
    const { username, password } = await request.json()

    if (!username || !password) {
      return NextResponse.json({ error: "Username and password are required" }, { status: 400 })
    }

    const user = await authenticateUser(username, password)

    if (!user) {
      return NextResponse.json({ error: "Invalid credentials" }, { status: 401 })
    }

    if (!user.emailVerified) {
      return NextResponse.json(
        {
          error: "Please verify your email address before signing in",
          requiresVerification: true,
        },
        { status: 401 },
      )
    }

    // Remove password from response
    const { password: _, ...userWithoutPassword } = user

    const response = NextResponse.json({
      success: true,
      user: userWithoutPassword,
    })

    // Set secure session cookie that expires in 30 days
    response.cookies.set("indus-session", JSON.stringify(userWithoutPassword), {
      httpOnly: false, // Allow client-side access for auth state
      secure: process.env.NODE_ENV === "production",
      sameSite: "lax",
      maxAge: 30 * 24 * 60 * 60, // 30 days
      path: "/",
    })

    return response
  } catch (error) {
    console.error("Login error:", error)
    return NextResponse.json({ error: "Internal server error" }, { status: 500 })
  }
}
