import { type NextRequest, NextResponse } from "next/server"
import { chargeUserForAdInteraction, client } from "@/lib/typesense"
//app/api/ads/track/route.ts
interface AdInteraction {
  adId: string
  action: "impression" | "click"
  timestamp: string
  userAgent?: string
  ip?: string
}

export async function POST(request: NextRequest) {
  try {
    const { adId, action } = await request.json()

    if (!adId || !action) {
      return NextResponse.json({ success: false, error: "Missing adId or action" }, { status: 400 })
    }

    // Log ad interaction for analytics
    const interaction: AdInteraction = {
      adId,
      action,
      timestamp: new Date().toISOString(),
      userAgent: request.headers.get("user-agent") || undefined,
      ip: request.headers.get("x-forwarded-for") || request.headers.get("x-real-ip") || undefined,
    }

    console.log("[v0] Ad interaction:", interaction)

    let isUserAd = false
    let charged = false

    try {
      // First try to find in user_ads collection
      const userAdResult = await client.collections("user_ads").documents(adId).retrieve()
      if (userAdResult) {
        isUserAd = true
        // Attempt to charge user for ad interaction
        charged = await chargeUserForAdInteraction(adId, action)

        if (!charged) {
          console.log(`[v0] User ad ${adId} auto-paused due to insufficient balance`)
        } else {
          console.log(`[v0] Successfully charged user for ${action} on user ad ${adId}`)

          // Update user ad statistics
          const currentStats = userAdResult
          const updates: any = {}

          if (action === "click") {
            updates.clicks = (currentStats.clicks || 0) + 1
          } else if (action === "impression") {
            updates.impressions = (currentStats.impressions || 0) + 1
          }

          if (Object.keys(updates).length > 0) {
            await client.collections("user_ads").documents(adId).update(updates)
            console.log(`[v0] Updated user ad ${adId} statistics:`, updates)
          }
        }
      }
    } catch (userAdError) {
      // Not a user ad, try admin ads
      try {
        const adminAdResult = await client.collections("ads_results").documents(adId).retrieve()
        if (adminAdResult) {
          console.log(`[v0] Admin ad interaction - no charge applied for ad ${adId}`)

          // Update admin ad statistics if needed
          if (action === "click") {
            console.log("[v0] Recording click for admin ad:", adId)
          }
        }
      } catch (adminAdError) {
        console.error(`[v0] Ad ${adId} not found in either collection:`, adminAdError)
      }
    }

    return NextResponse.json({
      success: true,
      charged: charged,
      isUserAd: isUserAd,
    })
  } catch (error) {
    console.error("[v0] Error tracking ad:", error)
    return NextResponse.json({ success: false, error: "Internal server error" }, { status: 500 })
  }
}
