import { type NextRequest, NextResponse } from "next/server"
import { searchWebsites, createWebsitesCollection, submitWebsite } from "@/lib/typesense"

export async function GET(request: NextRequest) {
  try {
    await createWebsitesCollection()

    const { searchParams } = new URL(request.url)
    const query = searchParams.get("q") || ""
    const status = searchParams.get("status") || ""
    const page = Number.parseInt(searchParams.get("page") || "1")
    const limit = Number.parseInt(searchParams.get("limit") || "50")

    console.log("[ADMIN WEBSITES] Fetching websites:", { query, status, page, limit })

    const result = await searchWebsites({
      query: query || undefined,
      status: status || undefined,
      page,
      limit,
    })

    const allWebsitesResult = await searchWebsites({ limit: 1000 })
    const allWebsites = allWebsitesResult.results

    const stats = {
      total: allWebsitesResult.total,
      pending: allWebsites.filter((w) => w.status === "pending").length,
      approved: allWebsites.filter((w) => w.status === "approved").length,
      rejected: allWebsites.filter((w) => w.status === "rejected").length,
    }

    console.log("[ADMIN WEBSITES] Stats calculated:", stats)

    return NextResponse.json({
      websites: result.results,
      total: result.total,
      hasMore: result.hasMore,
      stats,
    })
  } catch (error) {
    console.error("[ADMIN WEBSITES] Error:", error)
    return NextResponse.json({ error: "Failed to fetch websites" }, { status: 500 })
  }
}

export async function POST(request: NextRequest) {
  try {
    await createWebsitesCollection()

    const body = await request.json()
    const { title, url, imageUrl, supportWebCrawl, supportImageCrawl } = body

    if (!title || !url) {
      return NextResponse.json({ error: "Title and URL are required" }, { status: 400 })
    }

    const website = {
      title,
      url,
      imageUrl: imageUrl || undefined,
      supportImageCrawl: supportImageCrawl || false,
      supportWebCrawl: supportWebCrawl !== false, // Default to true
      ownerConfirmed: true, // Admin submissions are pre-confirmed
    }

    const result = await submitWebsite(website)
    console.log("[ADMIN WEBSITES] Created website:", result)

    return NextResponse.json({ success: true, website: result })
  } catch (error) {
    console.error("[ADMIN WEBSITES] Error creating website:", error)
    return NextResponse.json({ error: "Failed to create website" }, { status: 500 })
  }
}
