import { type NextRequest, NextResponse } from "next/server"
import { cookies } from "next/headers"
import { getAdminById } from "@/lib/typesense"
import { submitWebsite } from "@/lib/typesense"

export async function POST(request: NextRequest) {
  try {
    // Check admin authentication
    const cookieStore = cookies()
    const adminSession = cookieStore.get("admin_session")

    if (!adminSession) {
      return NextResponse.json({ error: "Unauthorized" }, { status: 401 })
    }

    const admin = await getAdminById(adminSession.value)
    if (!admin) {
      return NextResponse.json({ error: "Invalid admin session" }, { status: 401 })
    }

    const formData = await request.formData()
    const file = formData.get("file") as File

    if (!file) {
      return NextResponse.json({ error: "No file provided" }, { status: 400 })
    }

    if (!file.name.endsWith(".csv")) {
      return NextResponse.json({ error: "File must be a CSV" }, { status: 400 })
    }

    // Read and parse CSV
    const text = await file.text()
    const lines = text.split("\n").filter((line) => line.trim())

    if (lines.length < 2) {
      return NextResponse.json({ error: "CSV must contain at least a header and one data row" }, { status: 400 })
    }

    const headers = lines[0].split(",").map((h) => h.trim().replace(/"/g, ""))
    const requiredHeaders = ["title", "url"]
    const missingHeaders = requiredHeaders.filter((h) => !headers.includes(h))

    if (missingHeaders.length > 0) {
      return NextResponse.json(
        {
          error: `Missing required columns: ${missingHeaders.join(", ")}`,
        },
        { status: 400 },
      )
    }

    let successCount = 0
    let failedCount = 0
    const errors: string[] = []

    // Process each row
    for (let i = 1; i < lines.length; i++) {
      const line = lines[i].trim()
      if (!line) continue

      try {
        // Parse CSV row (simple parsing - doesn't handle quoted commas)
        const values = line.split(",").map((v) => v.trim().replace(/"/g, ""))
        const rowData: any = {}

        headers.forEach((header, index) => {
          rowData[header] = values[index] || ""
        })

        // Validate required fields
        if (!rowData.title || !rowData.url) {
          errors.push(`Row ${i + 1}: Missing title or URL`)
          failedCount++
          continue
        }

        // Validate URL format
        try {
          new URL(rowData.url)
        } catch {
          errors.push(`Row ${i + 1}: Invalid URL format`)
          failedCount++
          continue
        }

        // Convert boolean strings
        const website = {
          title: rowData.title,
          url: rowData.url,
          imageUrl: rowData.imageUrl || undefined,
          supportWebCrawl:
            rowData.supportWebCrawl === "true" || rowData.supportWebCrawl === "1" || !rowData.supportWebCrawl,
          supportImageCrawl: rowData.supportImageCrawl === "true" || rowData.supportImageCrawl === "1",
          ownerConfirmed: rowData.ownerConfirmed === "true" || rowData.ownerConfirmed === "1",
          notes: rowData.notes || undefined,
        }

        // Submit website (auto-approved for admin uploads)
        await submitWebsite({
          ...website,
          status: "approved" as const,
          reviewedAt: new Date().toISOString(),
          reviewedBy: admin.username,
        })

        successCount++
      } catch (error) {
        console.error(`Error processing row ${i + 1}:`, error)
        errors.push(`Row ${i + 1}: ${error instanceof Error ? error.message : "Processing failed"}`)
        failedCount++
      }
    }

    return NextResponse.json({
      success: successCount,
      failed: failedCount,
      errors: errors,
      message: `Processed ${successCount + failedCount} rows: ${successCount} successful, ${failedCount} failed`,
    })
  } catch (error) {
    console.error("Bulk upload error:", error)
    return NextResponse.json(
      {
        error: "Upload processing failed",
      },
      { status: 500 },
    )
  }
}
