import { type NextRequest, NextResponse } from "next/server"
import typesenseClient from "@/lib/typesense"

export async function GET(request: NextRequest) {
  try {
    const { searchParams } = new URL(request.url)
    const query = searchParams.get("search") || searchParams.get("q") || "*"

    const searchParameters = {
      q: query,
      query_by: query === "*" ? "title" : "title,description,keywords",
      sort_by: "_text_match:desc",
      per_page: 100,
    }

    const searchResults = await typesenseClient.collections("web_results").documents().search(searchParameters)

    const results =
      searchResults.hits?.map((hit: any) => ({
        id: hit.document.id,
        title: hit.document.title,
        description: hit.document.description,
        url: hit.document.url,
        favicon: hit.document.favicon,
        date: hit.document.date,
        keywords: hit.document.keywords,
        safesearch: hit.document.safesearch,
        maindomain: hit.document.maindomain,
      })) || []

    return NextResponse.json({ results })
  } catch (error) {
    console.error("Error fetching web results:", error)
    return NextResponse.json({ results: [] })
  }
}

export async function POST(request: NextRequest) {
  try {
    const data = await request.json()
    const { title, description, url, keywords, safesearch = true, maindomain = false } = data

    if (!title || !description || !url) {
      return NextResponse.json({ success: false, error: "Title, description, and URL are required" }, { status: 400 })
    }

    const document = {
      id: `${Date.now()}-${Math.random().toString(36).substr(2, 9)}`,
      title,
      description,
      url,
      date: new Date().toISOString().split("T")[0],
      keywords: keywords || [],
      safesearch: Boolean(safesearch),
      maindomain: Boolean(maindomain),
      favicon: `https://www.google.com/s2/favicons?domain=${new URL(url).hostname}&sz=32`,
    }

    await typesenseClient.collections("web_results").documents().create(document)

    return NextResponse.json({
      success: true,
      message: "Web result created successfully",
    })
  } catch (error) {
    console.error("Create web result error:", error)
    return NextResponse.json(
      {
        success: false,
        error: error instanceof Error ? error.message : "Unknown error",
      },
      { status: 500 },
    )
  }
}
