import { type NextRequest, NextResponse } from "next/server"
import { writeFile, mkdir } from "fs/promises"
import { join } from "path"
import { existsSync } from "fs"
import { setSiteSetting } from "@/lib/typesense"

const ALLOWED_TYPES = {
  favicon: ["image/x-icon", "image/vnd.microsoft.icon", "image/png", "image/jpeg"],
  logo: ["image/png", "image/jpeg", "image/svg+xml", "image/webp"],
}

const MAX_FILE_SIZE = 5 * 1024 * 1024 // 5MB

export async function POST(request: NextRequest) {
  try {
    const formData = await request.formData()
    const file = formData.get("file") as File
    const type = formData.get("type") as string // "favicon" or "logo"
    const adminId = formData.get("adminId") as string

    if (!file) {
      return NextResponse.json({ error: "No file provided" }, { status: 400 })
    }

    if (!type || !["favicon", "logo"].includes(type)) {
      return NextResponse.json({ error: "Invalid file type. Must be 'favicon' or 'logo'" }, { status: 400 })
    }

    if (!adminId) {
      return NextResponse.json(
        { error: "Admin authentication required" },
        {
          status: 401```tsx file="app/api/admin/upload/route.ts"
import { NextRequest, NextResponse } from "next/server"
import { writeFile, mkdir } from "fs/promises"
import { join } from "path"
import { existsSync } from "fs"
import { setSiteSetting } from "@/lib/typesense"

const ALLOWED_TYPES = {
  favicon: ["image/x-icon", "image/vnd.microsoft.icon", "image/png", "image/jpeg"],
  logo: ["image/png", "image/jpeg", "image/svg+xml", "image/webp"],
}

const MAX_FILE_SIZE = 5 * 1024 * 1024 // 5MB

export async function POST(request: NextRequest) {
  try {
    const formData = await request.formData()
    const file = formData.get("file") as File
    const type = formData.get("type") as string // "favicon" or "logo"
    const adminId = formData.get("adminId") as string

    if (!file) {
      return NextResponse.json({ error: "No file provided" }, { status: 400 })
    }

    if (!type || !["favicon", "logo"].includes(type)) {
      return NextResponse.json({ error: "Invalid file type. Must be 'favicon' or 'logo'" }, { status: 400 })
    }

    if (!adminId) {
      return NextResponse.json({ error: "Admin authentication required" }, { status: 401 })
    }

    // Validate file type
    if (!ALLOWED_TYPES[type].includes(file.type)) {
      return NextResponse.json(
        {
          error: \`Invalid file format for ${type}. Allowed: ${ALLOWED_TYPES[type].join(", ")}`,
        },
        { status: 400 },
      )
    }

    // Validate file size
    if (file.size > MAX_FILE_SIZE) {
      return NextResponse.json(
        { error: `File too large. Maximum size: ${MAX_FILE_SIZE / 1024 / 1024}MB` },
        { status: 400 },
      )
    }

    // Create uploads directory if it doesn't exist
    const uploadsDir = join(process.cwd(), "public", "uploads")
    if (!existsSync(uploadsDir)) {
      await mkdir(uploadsDir, { recursive: true })
    }

    // Generate unique filename
    const timestamp = Date.now()
    const extension = file.name.split(".").pop()
    const filename = `${type}-${timestamp}.${extension}`
    const filepath = join(uploadsDir, filename)
    const publicPath = `/uploads/${filename}`

    // Save file
    const bytes = await file.arrayBuffer()
    const buffer = Buffer.from(bytes)
    await writeFile(filepath, buffer)

    // Update site settings
    const settingKey = type === "favicon" ? "favicon" : "logo"
    await setSiteSetting(settingKey, publicPath, "file", "appearance", adminId)

    console.log(`[UPLOAD] Successfully uploaded ${type}: ${publicPath}`)

    return NextResponse.json({
      success: true,
      filename,
      path: publicPath,
      size: file.size,
      type: file.type,
    })
  } catch (error) {
    console.error("[UPLOAD] Error uploading file:", error)
    return NextResponse.json({ error: "Failed to upload file" }, { status: 500 })
  }
}

export async function DELETE(request: NextRequest) {
  try {
    const { searchParams } = new URL(request.url)
    const type = searchParams.get("type")
    const adminId = searchParams.get("adminId")

    if (!type || !["favicon", "logo"].includes(type)) {
      return NextResponse.json({ error: "Invalid file type" }, { status: 400 })
    }

    if (!adminId) {
      return NextResponse.json({ error: "Admin authentication required" }, { status: 401 })
    }

    // Reset to default
    const defaultPath = type === "favicon" ? "/favicon.ico" : "/logo.png"
    const settingKey = type === "favicon" ? "favicon" : "logo"

    await setSiteSetting(settingKey, defaultPath, "file", "appearance", adminId)

    console.log(`[UPLOAD] Reset ${type} to default: ${defaultPath}`)

    return NextResponse.json({
      success: true,
      path: defaultPath,
    })
  } catch (error) {
    console.error("[UPLOAD] Error resetting file:", error)
    return NextResponse.json({ error: "Failed to reset file" }, { status: 500 })
  }
}
