import { NextResponse } from "next/server"
import typesenseClient from "@/lib/typesense"

export async function GET() {
  try {
    let collections: Array<{ name: string; numDocuments: number; createdAt: string }> = []
    let typesenseStatus: "connected" | "disconnected" = "disconnected"

    try {
      const collectionsResponse = await typesenseClient.collections().retrieve()
      typesenseStatus = "connected"

      collections = await Promise.all(
        collectionsResponse.map(async (collection: any) => {
          try {
            const collectionInfo = await typesenseClient.collections(collection.name).retrieve()
            return {
              name: collection.name,
              numDocuments: collectionInfo.num_documents || 0,
              createdAt: new Date(collectionInfo.created_at * 1000).toISOString(),
            }
          } catch (error) {
            return {
              name: collection.name,
              numDocuments: 0,
              createdAt: new Date().toISOString(),
            }
          }
        }),
      )
    } catch (error) {
      console.error("Typesense connection error:", error)
    }

    const youtubeStatus = process.env.YOUTUBE_API_KEY ? "configured" : "not-configured"

    return NextResponse.json({
      collections,
      typesenseStatus,
      youtubeStatus,
    })
  } catch (error) {
    console.error("Error fetching system info:", error)
    return NextResponse.json(
      {
        collections: [],
        typesenseStatus: "disconnected",
        youtubeStatus: "not-configured",
      },
      { status: 500 },
    )
  }
}
