import { NextResponse } from "next/server"
import typesenseClient from "@/lib/typesense"

export async function GET() {
  try {
    console.log("[v0] Fetching admin stats...")

    const stats = {
      totalCollections: 0,
      webResults: 0,
      imageResults: 0,
      videoResults: 0,
      totalUsers: 0,
      totalRevenue: 0,
      totalAds: 0,
      lastUpdated: new Date().toISOString(),
    }

    try {
      const collections = await typesenseClient.collections().retrieve()
      stats.totalCollections = collections.length

      // Get document counts for each collection
      for (const collection of collections) {
        try {
          const collectionInfo = await typesenseClient.collections(collection.name).retrieve()

          if (collection.name === "web_results") {
            stats.webResults = collectionInfo.num_documents || 0
          } else if (collection.name === "image_results") {
            stats.imageResults = collectionInfo.num_documents || 0
          } else if (collection.name === "users") {
            stats.totalUsers = collectionInfo.num_documents || 0
          } else if (collection.name === "user_ads") {
            stats.totalAds = collectionInfo.num_documents || 0
          }
        } catch (error) {
          console.error(`[v0] Error getting stats for collection ${collection.name}:`, error)
        }
      }

      try {
        const transactionResults = await typesenseClient.collections("wallet_transactions").documents().search({
          q: "*",
          query_by: "description",
          filter_by: "type:=credit",
          per_page: 250,
        })

        let totalRevenue = 0
        if (transactionResults.hits) {
          for (const hit of transactionResults.hits) {
            const transaction = hit.document as any
            // Only count transactions with Razorpay payment IDs (actual payments)
            if (transaction.amount && typeof transaction.amount === "number" && transaction.razorpayPaymentId) {
              totalRevenue += transaction.amount
            }
          }
        }
        stats.totalRevenue = totalRevenue
      } catch (revenueError) {
        console.error("[v0] Error calculating revenue:", revenueError)
        stats.totalRevenue = 0
      }
    } catch (connectionError) {
      console.error("[v0] Typesense connection failed, returning default stats:", connectionError)
      // Return default stats instead of throwing error
    }

    console.log("[v0] Stats fetched successfully:", stats)
    return NextResponse.json(stats)
  } catch (error) {
    console.error("[v0] Error fetching admin stats:", error)
    return NextResponse.json({
      totalCollections: 0,
      webResults: 0,
      imageResults: 0,
      videoResults: 0,
      totalUsers: 0,
      totalRevenue: 0,
      totalAds: 0,
      lastUpdated: new Date().toISOString(),
      error: "Connection failed - showing default values",
    })
  }
}
