import { type NextRequest, NextResponse } from "next/server"
import { getAllSiteSettings, setSiteSetting } from "@/lib/typesense"

export async function GET() {
  try {
    const settings = await getAllSiteSettings()
    return NextResponse.json(settings)
  } catch (error) {
    console.error("[ADMIN] Error fetching site settings:", error)
    return NextResponse.json({ error: "Failed to fetch site settings" }, { status: 500 })
  }
}

export async function POST(request: NextRequest) {
  try {
    const data = await request.json()
    const { adminId, ...settings } = data

    if (!adminId) {
      return NextResponse.json({ error: "Admin authentication required" }, { status: 401 })
    }

    // Save each setting individually
    const settingsMap = [
      { key: "siteName", value: settings.siteName, type: "text", category: "general" },
      { key: "siteTitle", value: settings.siteTitle, type: "text", category: "seo" },
      { key: "siteDescription", value: settings.siteDescription, type: "text", category: "seo" },
      { key: "siteKeywords", value: settings.siteKeywords, type: "text", category: "seo" },
      { key: "themeColor", value: settings.themeColor, type: "color", category: "appearance" },
      { key: "favicon", value: settings.favicon, type: "file", category: "appearance" },
      { key: "logo", value: settings.logo, type: "file", category: "appearance" },
      { key: "welcomeMessage", value: settings.welcomeMessage, type: "text", category: "homepage" },
      { key: "popularSearches", value: settings.popularSearches, type: "json", category: "homepage" },
      { key: "showSubmitWebsite", value: settings.showSubmitWebsite, type: "boolean", category: "homepage" },
    ]

    for (const setting of settingsMap) {
      await setSiteSetting(setting.key, setting.value, setting.type, setting.category, adminId)
    }

    console.log("[ADMIN] Site settings updated successfully")
    return NextResponse.json({ success: true })
  } catch (error) {
    console.error("[ADMIN] Error saving site settings:", error)
    return NextResponse.json({ error: "Failed to save site settings" }, { status: 500 })
  }
}
