import { type NextRequest, NextResponse } from "next/server"
import typesenseClient from "@/lib/typesense"

// Dummy data for popular websites
const webResultsData = [
  {
    id: "1",
    title: "Google - Search the world's information",
    description:
      "Search the world's information, including webpages, images, videos and more. Google has many special features to help you find exactly what you're looking for.",
    url: "https://www.google.com",
    date: "2024-01-15",
    keywords: ["search", "google", "web", "information"],
    safesearch: true,
    maindomain: true,
    favicon: "https://www.google.com/s2/favicons?domain=google.com&sz=32",
  },
  {
    id: "2",
    title: "Facebook - Connect with friends and the world around you",
    description:
      "Create an account or log into Facebook. Connect with friends, family and other people you know. Share photos and videos, send messages and get updates.",
    url: "https://www.facebook.com",
    date: "2024-01-14",
    keywords: ["social", "facebook", "friends", "connect"],
    safesearch: true,
    maindomain: true,
    favicon: "https://www.google.com/s2/favicons?domain=facebook.com&sz=32",
  },
  {
    id: "3",
    title: "Instagram - Discover what's happening right now",
    description:
      "Create an account or log in to Instagram - A simple, fun & creative way to capture, edit & share photos, videos & messages with friends & family.",
    url: "https://www.instagram.com",
    date: "2024-01-13",
    keywords: ["photos", "instagram", "social", "sharing"],
    safesearch: true,
    maindomain: true,
    favicon: "https://www.google.com/s2/favicons?domain=instagram.com&sz=32",
  },
  {
    id: "4",
    title: "YouTube - Broadcast Yourself",
    description:
      "Enjoy the videos and music you love, upload original content, and share it all with friends, family, and the world on YouTube.",
    url: "https://www.youtube.com",
    date: "2024-01-12",
    keywords: ["videos", "youtube", "music", "entertainment"],
    safesearch: true,
    maindomain: true,
    favicon: "https://www.google.com/s2/favicons?domain=youtube.com&sz=32",
  },
  {
    id: "5",
    title: "Twitter - What's happening?",
    description:
      "From breaking news and entertainment to sports and politics, get the full story with all the live commentary.",
    url: "https://www.twitter.com",
    date: "2024-01-11",
    keywords: ["twitter", "news", "social", "updates"],
    safesearch: true,
    maindomain: true,
    favicon: "https://www.google.com/s2/favicons?domain=twitter.com&sz=32",
  },
  {
    id: "6",
    title: "LinkedIn - The World's Largest Professional Network",
    description:
      "750+ million members | Manage your professional identity. Build and engage with your professional network. Access knowledge, insights and opportunities.",
    url: "https://www.linkedin.com",
    date: "2024-01-10",
    keywords: ["professional", "linkedin", "network", "career"],
    safesearch: true,
    maindomain: true,
    favicon: "https://www.google.com/s2/favicons?domain=linkedin.com&sz=32",
  },
  {
    id: "7",
    title: "Amazon - Earth's Most Customer-Centric Company",
    description:
      "Free shipping on millions of items. Get the best of Shopping and Entertainment with Prime. Enjoy low prices and great deals on the largest selection.",
    url: "https://www.amazon.com",
    date: "2024-01-09",
    keywords: ["shopping", "amazon", "ecommerce", "products"],
    safesearch: true,
    maindomain: true,
    favicon: "https://www.google.com/s2/favicons?domain=amazon.com&sz=32",
  },
  {
    id: "8",
    title: "Netflix - Watch TV Shows Online, Watch Movies Online",
    description:
      "Watch Netflix movies & TV shows online or stream right to your smart TV, game console, PC, Mac, mobile, tablet and more.",
    url: "https://www.netflix.com",
    date: "2024-01-08",
    keywords: ["streaming", "netflix", "movies", "tv shows"],
    safesearch: true,
    maindomain: true,
    favicon: "https://www.google.com/s2/favicons?domain=netflix.com&sz=32",
  },
  {
    id: "9",
    title: "GitHub - Where the world builds software",
    description:
      "GitHub is where over 100 million developers shape the future of software, together. Contribute to the open source community, manage Git repositories.",
    url: "https://www.github.com",
    date: "2024-01-07",
    keywords: ["code", "github", "developers", "open source"],
    safesearch: true,
    maindomain: true,
    favicon: "https://www.google.com/s2/favicons?domain=github.com&sz=32",
  },
  {
    id: "10",
    title: "Stack Overflow - Where Developers Learn, Share, & Build Careers",
    description:
      "Stack Overflow is the largest, most trusted online community for developers to learn, share their programming knowledge, and build their careers.",
    url: "https://stackoverflow.com",
    date: "2024-01-06",
    keywords: ["programming", "stackoverflow", "developers", "questions"],
    safesearch: true,
    maindomain: true,
    favicon: "https://www.google.com/s2/favicons?domain=stackoverflow.com&sz=32",
  },
]

const imageResultsData = [
  {
    id: "img1",
    title: "Google Logo",
    url: "https://www.google.com",
    imgurl: "https://www.google.com/images/branding/googlelogo/1x/googlelogo_color_272x92dp.png",
    keywords: ["google", "logo", "brand"],
    date: "2024-01-15",
    safesearch: true,
    favicon: "https://www.google.com/s2/favicons?domain=google.com&sz=32",
  },
  {
    id: "img2",
    title: "Facebook Logo",
    url: "https://www.facebook.com",
    imgurl: "https://upload.wikimedia.org/wikipedia/commons/5/51/Facebook_f_logo_%282019-2017%29.svg",
    keywords: ["facebook", "logo", "social"],
    date: "2024-01-14",
    safesearch: true,
    favicon: "https://www.google.com/s2/favicons?domain=facebook.com&sz=32",
  },
  {
    id: "img3",
    title: "Instagram Logo",
    url: "https://www.instagram.com",
    imgurl: "https://upload.wikimedia.org/wikipedia/commons/a/a5/Instagram_icon.png",
    keywords: ["instagram", "logo", "camera"],
    date: "2024-01-13",
    safesearch: true,
    favicon: "https://www.google.com/s2/favicons?domain=instagram.com&sz=32",
  },
  {
    id: "img4",
    title: "YouTube Logo",
    url: "https://www.youtube.com",
    imgurl: "https://upload.wikimedia.org/wikipedia/commons/4/42/YouTube_icon_%282013-2017%29.png",
    keywords: ["youtube", "logo", "video"],
    date: "2024-01-12",
    safesearch: true,
    favicon: "https://www.google.com/s2/favicons?domain=youtube.com&sz=32",
  },
  {
    id: "img5",
    title: "Twitter Logo",
    url: "https://www.twitter.com",
    imgurl: "https://upload.wikimedia.org/wikipedia/commons/6/6f/Logo_of_Twitter.svg",
    keywords: ["twitter", "logo", "bird"],
    date: "2024-01-11",
    safesearch: true,
    favicon: "https://www.google.com/s2/favicons?domain=twitter.com&sz=32",
  },
  {
    id: "img6",
    title: "LinkedIn Logo",
    url: "https://www.linkedin.com",
    imgurl: "https://upload.wikimedia.org/wikipedia/commons/c/ca/LinkedIn_logo_initials.png",
    keywords: ["linkedin", "logo", "professional"],
    date: "2024-01-10",
    safesearch: true,
    favicon: "https://www.google.com/s2/favicons?domain=linkedin.com&sz=32",
  },
]

export async function POST(request: NextRequest) {
  try {
    const { type } = await request.json()

    if (type === "web" || type === "all") {
      // Create web_results collection schema
      const webSchema = {
        name: "web_results",
        fields: [
          { name: "title", type: "string" },
          { name: "description", type: "string" },
          { name: "url", type: "string" },
          { name: "date", type: "string", optional: true },
          { name: "keywords", type: "string[]", optional: true },
          { name: "safesearch", type: "bool" },
          { name: "maindomain", type: "bool" },
          { name: "favicon", type: "string", optional: true },
        ],
      }

      try {
        await typesenseClient.collections("web_results").delete()
      } catch (e) {
        // Collection might not exist
      }

      await typesenseClient.collections().create(webSchema)

      // Insert web results data
      for (const doc of webResultsData) {
        await typesenseClient.collections("web_results").documents().create(doc)
      }
    }

    if (type === "images" || type === "all") {
      // Create image_results collection schema
      const imageSchema = {
        name: "image_results",
        fields: [
          { name: "title", type: "string" },
          { name: "url", type: "string" },
          { name: "imgurl", type: "string" },
          { name: "keywords", type: "string[]", optional: true },
          { name: "date", type: "string", optional: true },
          { name: "safesearch", type: "bool" },
          { name: "favicon", type: "string", optional: true },
        ],
      }

      try {
        await typesenseClient.collections("image_results").delete()
      } catch (e) {
        // Collection might not exist
      }

      await typesenseClient.collections().create(imageSchema)

      // Insert image results data
      for (const doc of imageResultsData) {
        await typesenseClient.collections("image_results").documents().create(doc)
      }
    }

    return NextResponse.json({
      success: true,
      message: `Successfully seeded ${type} data`,
      webResults: type === "web" || type === "all" ? webResultsData.length : 0,
      imageResults: type === "images" || type === "all" ? imageResultsData.length : 0,
    })
  } catch (error) {
    console.error("Seeding error:", error)
    return NextResponse.json(
      {
        success: false,
        error: error instanceof Error ? error.message : "Unknown error",
      },
      { status: 500 },
    )
  }
}
