import { type NextRequest, NextResponse } from "next/server"
import typesenseClient from "@/lib/typesense"

export async function GET(request: NextRequest) {
  try {
    const { searchParams } = new URL(request.url)
    const query = searchParams.get("search") || searchParams.get("q") || "*"

    const searchParameters = {
      q: query,
      query_by: query === "*" ? "title" : "title,keywords",
      sort_by: "_text_match:desc",
      per_page: 100,
    }

    const searchResults = await typesenseClient.collections("image_results").documents().search(searchParameters)

    const results =
      searchResults.hits?.map((hit: any) => ({
        id: hit.document.id,
        title: hit.document.title,
        url: hit.document.url,
        imgurl: hit.document.imgurl,
        keywords: hit.document.keywords,
        favicon: hit.document.favicon,
        date: hit.document.date,
        safesearch: hit.document.safesearch,
      })) || []

    return NextResponse.json({ results })
  } catch (error) {
    console.error("Error fetching image results:", error)
    return NextResponse.json({ results: [] })
  }
}

export async function POST(request: NextRequest) {
  try {
    const data = await request.json()
    const { title, url, imgurl, keywords, safesearch = true } = data

    if (!title || !url || !imgurl) {
      return NextResponse.json({ success: false, error: "Title, URL, and image URL are required" }, { status: 400 })
    }

    const document = {
      id: `${Date.now()}-${Math.random().toString(36).substr(2, 9)}`,
      title,
      url,
      imgurl,
      keywords: keywords || [],
      date: new Date().toISOString().split("T")[0],
      safesearch: Boolean(safesearch),
      favicon: `https://www.google.com/s2/favicons?domain=${new URL(url).hostname}&sz=32`,
    }

    await typesenseClient.collections("image_results").documents().create(document)

    return NextResponse.json({
      success: true,
      message: "Image result created successfully",
    })
  } catch (error) {
    console.error("Create image result error:", error)
    return NextResponse.json(
      {
        success: false,
        error: error instanceof Error ? error.message : "Unknown error",
      },
      { status: 500 },
    )
  }
}
