import { type NextRequest, NextResponse } from "next/server"
import typesenseClient from "@/lib/typesense"

function parseCSVLine(line: string): string[] {
  const result = []
  let current = ""
  let inQuotes = false

  for (let i = 0; i < line.length; i++) {
    const char = line[i]

    if (char === '"') {
      if (inQuotes && line[i + 1] === '"') {
        current += '"'
        i++ // Skip next quote
      } else {
        inQuotes = !inQuotes
      }
    } else if (char === "," && !inQuotes) {
      result.push(current.trim())
      current = ""
    } else {
      current += char
    }
  }

  result.push(current.trim())
  return result
}

export async function POST(request: NextRequest) {
  try {
    const formData = await request.formData()
    const file = formData.get("file") as File

    if (!file) {
      return NextResponse.json({ error: "No file provided" }, { status: 400 })
    }

    const text = await file.text()
    const lines = text.split("\n").filter((line) => line.trim())

    if (lines.length < 2) {
      return NextResponse.json({ error: "CSV file must have at least a header and one data row" }, { status: 400 })
    }

    const headers = parseCSVLine(lines[0]).map((h) => h.toLowerCase().trim())
    const documents = []
    const errors = []

    console.log("[v0] CSV Headers found:", headers)

    for (let i = 1; i < lines.length; i++) {
      const values = parseCSVLine(lines[i])

      if (values.length !== headers.length) {
        errors.push(`Row ${i + 1}: Column count mismatch (expected ${headers.length}, got ${values.length})`)
        continue
      }

      const doc: any = {
        id: `${Date.now()}-${i}-${Math.random().toString(36).substr(2, 9)}`,
        date: new Date().toISOString().split("T")[0],
      }

      headers.forEach((header, index) => {
        const value = values[index]?.trim() || ""

        switch (header) {
          case "keywords":
            doc[header] = value
              ? value
                  .split(";")
                  .map((k) => k.trim())
                  .filter((k) => k)
              : []
            break
          case "safesearch":
            doc[header] = value.toLowerCase() === "true"
            break
          default:
            doc[header] = value
        }
      })

      const requiredFields = ["title", "url", "imgurl"]
      const missingFields = requiredFields.filter((field) => !doc[field] || doc[field].trim() === "")

      if (missingFields.length > 0) {
        errors.push(`Row ${i + 1}: Missing required fields: ${missingFields.join(", ")}`)
        continue
      }

      try {
        new URL(doc.url)
        new URL(doc.imgurl)
        doc.favicon = `https://www.google.com/s2/favicons?domain=${new URL(doc.url).hostname}&sz=32`
        documents.push(doc)
        console.log("[v0] Valid document added:", { title: doc.title, url: doc.url, imgurl: doc.imgurl })
      } catch (urlError) {
        errors.push(`Row ${i + 1}: Invalid URL format in url or imgurl fields`)
      }
    }

    console.log("[v0] Processing complete:", {
      totalRows: lines.length - 1,
      validDocuments: documents.length,
      errors: errors.length,
    })

    if (documents.length === 0) {
      return NextResponse.json(
        {
          error: "No valid documents found in CSV",
          details: errors.slice(0, 10), // Show first 10 errors
          totalErrors: errors.length,
        },
        { status: 400 },
      )
    }

    const importResult = await typesenseClient.collections("image_results").documents().import(documents, {
      action: "create",
    })

    const successCount = importResult.filter((result: any) => result.success === true).length
    const errorCount = documents.length - successCount

    return NextResponse.json({
      success: true,
      message: `Successfully imported ${successCount} image results${errorCount > 0 ? ` (${errorCount} failed)` : ""}`,
      count: successCount,
      errors: errorCount,
      validationErrors: errors.length > 0 ? errors.slice(0, 5) : undefined,
    })
  } catch (error) {
    console.error("Bulk upload error:", error)
    return NextResponse.json(
      {
        error: error instanceof Error ? error.message : "Bulk upload failed",
      },
      { status: 500 },
    )
  }
}
