import { type NextRequest, NextResponse } from "next/server"

interface HomepageSettings {
  popularSearches: string[]
  welcomeMessage: string
  showSubmitWebsite: boolean
}

// Default settings
const defaultSettings: HomepageSettings = {
  popularSearches: ["Technology", "News", "Weather", "Sports", "Entertainment"],
  welcomeMessage: "Fast, comprehensive search across the web",
  showSubmitWebsite: true,
}

// In a real app, this would be stored in a database
let homepageSettings: HomepageSettings = { ...defaultSettings }

export async function GET() {
  try {
    return NextResponse.json(homepageSettings)
  } catch (error) {
    console.error("[HOMEPAGE SETTINGS] Error fetching settings:", error)
    return NextResponse.json(defaultSettings)
  }
}

export async function POST(request: NextRequest) {
  try {
    const body = await request.json()

    // Validate the settings
    const settings: HomepageSettings = {
      popularSearches: Array.isArray(body.popularSearches)
        ? body.popularSearches.slice(0, 5).filter((search: any) => typeof search === "string" && search.trim())
        : defaultSettings.popularSearches,
      welcomeMessage:
        typeof body.welcomeMessage === "string"
          ? body.welcomeMessage.trim() || defaultSettings.welcomeMessage
          : defaultSettings.welcomeMessage,
      showSubmitWebsite:
        typeof body.showSubmitWebsite === "boolean" ? body.showSubmitWebsite : defaultSettings.showSubmitWebsite,
    }

    // Update the settings
    homepageSettings = settings

    console.log("[HOMEPAGE SETTINGS] Settings updated:", settings)

    return NextResponse.json({ success: true, settings })
  } catch (error) {
    console.error("[HOMEPAGE SETTINGS] Error saving settings:", error)
    return NextResponse.json({ error: "Failed to save homepage settings" }, { status: 500 })
  }
}
