import { type NextRequest, NextResponse } from "next/server"
import { client } from "@/lib/typesense"

export async function POST(request: NextRequest) {
  try {
    const { type, data } = await request.json()

    if (!type || !data || !Array.isArray(data)) {
      return NextResponse.json({ error: "Invalid request format" }, { status: 400 })
    }

    // Check if environment variables are configured
    if (!process.env.TYPESENSE_HOST || !process.env.TYPESENSE_API_KEY) {
      return NextResponse.json(
        {
          error: "Typesense not configured. Please add TYPESENSE_HOST and TYPESENSE_API_KEY to environment variables.",
        },
        { status: 400 },
      )
    }

    const collectionName = type === "web" ? "web_results" : "image_results"

    // Prepare documents with auto-generated IDs and timestamps
    const documents = data.map((item, index) => ({
      id: `${Date.now()}-${index}`,
      date: new Date().toISOString().split("T")[0],
      favicon:
        item.favicon ||
        `https://www.google.com/s2/favicons?domain=${new URL(item.url || "https://example.com").hostname}&sz=32`,
      ...item,
    }))

    // Try to create collection first (in case it doesn't exist)
    try {
      const schema =
        type === "web"
          ? {
              name: collectionName,
              fields: [
                { name: "title", type: "string" },
                { name: "description", type: "string", optional: true },
                { name: "url", type: "string" },
                { name: "keywords", type: "string[]", optional: true },
                { name: "safesearch", type: "bool", optional: true },
                { name: "date", type: "string", optional: true },
                { name: "favicon", type: "string", optional: true },
                { name: "maindomain", type: "bool", optional: true },
              ],
            }
          : {
              name: collectionName,
              fields: [
                { name: "title", type: "string" },
                { name: "url", type: "string" },
                { name: "imgurl", type: "string" },
                { name: "keywords", type: "string[]", optional: true },
                { name: "favicon", type: "string", optional: true },
                { name: "date", type: "string", optional: true },
                { name: "safesearch", type: "bool", optional: true },
              ],
            }

      await client.collections().create(schema)
    } catch (error) {
      // Collection might already exist, continue
      console.log(`Collection ${collectionName} might already exist:`, error)
    }

    // Import documents
    const importResult = await client.collections(collectionName).documents().import(documents, {
      action: "create",
    })

    // Count successful imports
    const successCount = importResult.filter((result: any) => result.success === true).length
    const errorCount = documents.length - successCount

    if (errorCount > 0) {
      console.warn(`${errorCount} documents failed to import`)
    }

    return NextResponse.json({
      success: true,
      message: `Successfully imported ${successCount} ${type} results${errorCount > 0 ? ` (${errorCount} failed)` : ""}`,
      count: successCount,
      errors: errorCount,
    })
  } catch (error) {
    console.error("Bulk upload error:", error)
    return NextResponse.json(
      {
        error: error instanceof Error ? error.message : "Bulk upload failed",
      },
      { status: 500 },
    )
  }
}
