import { type NextRequest, NextResponse } from "next/server"
import { createAdminUser, checkAdminUsernameAvailability } from "@/lib/typesense"

export async function POST(request: NextRequest) {
  try {
    // Check if admin registration is allowed
    if (process.env.ALLOW_ADMIN_REGISTER !== "true") {
      return NextResponse.json({ message: "Admin registration is disabled" }, { status: 403 })
    }

    const { username, password } = await request.json()

    if (!username || !password) {
      return NextResponse.json({ message: "Username and password are required" }, { status: 400 })
    }

    if (password.length < 6) {
      return NextResponse.json({ message: "Password must be at least 6 characters long" }, { status: 400 })
    }

    // Check if username is available
    const isAvailable = await checkAdminUsernameAvailability(username)
    if (!isAvailable) {
      return NextResponse.json({ message: "Username already exists" }, { status: 400 })
    }

    // Create admin user
    await createAdminUser(username, password)

    return NextResponse.json({ message: "Admin account created successfully" })
  } catch (error) {
    console.error("Admin registration error:", error)
    return NextResponse.json({ message: "Registration failed" }, { status: 500 })
  }
}
