import { type NextRequest, NextResponse } from "next/server"
import { authenticateAdmin } from "@/lib/typesense"
import { cookies } from "next/headers"

export async function POST(request: NextRequest) {
  try {
    const { username, password } = await request.json()

    if (!username || !password) {
      return NextResponse.json({ message: "Username and password are required" }, { status: 400 })
    }

    const admin = await authenticateAdmin(username, password)

    if (!admin) {
      return NextResponse.json({ message: "Invalid credentials" }, { status: 401 })
    }

    // Set admin session cookie
    const cookieStore = cookies()
    cookieStore.set("admin_session", admin.id, {
      httpOnly: true,
      secure: process.env.NODE_ENV === "production",
      sameSite: "lax",
      maxAge: 24 * 60 * 60, // 24 hours
    })

    return NextResponse.json({
      message: "Login successful",
      admin: {
        id: admin.id,
        username: admin.username,
        email: admin.email,
        createdAt: admin.createdAt,
        lastLoginAt: admin.lastLoginAt,
      },
    })
  } catch (error) {
    console.error("Admin login error:", error)
    return NextResponse.json({ message: "Login failed" }, { status: 500 })
  }
}
