import { type NextRequest, NextResponse } from "next/server"
import typesenseClient, { createAdsCollection } from "@/lib/typesense"

export async function GET(request: NextRequest) {
  try {
    if (!process.env.TYPESENSE_API_KEY || !process.env.TYPESENSE_HOST) {
      return NextResponse.json(
        {
          success: false,
          error: "Typesense is not configured. Please add environment variables in Project Settings.",
        },
        { status: 400 },
      )
    }

    const { searchParams } = new URL(request.url)
    const range = searchParams.get("range") || "7d"
    const sort = searchParams.get("sort") || "impressions"

    // Ensure ads collection exists
    await createAdsCollection()

    // Fetch all ads
    const searchResults = await typesenseClient.collections("ads_results").documents().search({
      q: "*",
      per_page: 250,
    })

    const ads = searchResults.hits?.map((hit: any) => hit.document) || []

    // Calculate analytics for each ad
    const analytics = ads.map((ad: any) => {
      const impressions = ad.monthlyShown || 0
      const clicks = Math.floor(impressions * 0.02) // Simulate 2% CTR for demo
      const ctr = impressions > 0 ? clicks / impressions : 0

      return {
        adId: ad.id,
        title: ad.title,
        impressions,
        clicks,
        ctr,
        dailyShown: ad.dailyShown || 0,
        monthlyShown: ad.monthlyShown || 0,
        dailyLimit: ad.dailyLimit || 100,
        monthlyLimit: ad.monthlyLimit || 3000,
        active: ad.active,
        keywords: ad.keywords || [],
      }
    })

    // Sort analytics
    analytics.sort((a, b) => {
      switch (sort) {
        case "clicks":
          return b.clicks - a.clicks
        case "ctr":
          return b.ctr - a.ctr
        case "title":
          return a.title.localeCompare(b.title)
        default: // impressions
          return b.impressions - a.impressions
      }
    })

    // Calculate summary
    const totalImpressions = analytics.reduce((sum, ad) => sum + ad.impressions, 0)
    const totalClicks = analytics.reduce((sum, ad) => sum + ad.clicks, 0)
    const averageCTR = totalImpressions > 0 ? totalClicks / totalImpressions : 0
    const activeAds = analytics.filter((ad) => ad.active).length
    const topPerformingAd = analytics.length > 0 ? analytics[0].title : ""

    const summary = {
      totalImpressions,
      totalClicks,
      averageCTR,
      activeAds,
      totalAds: analytics.length,
      topPerformingAd,
    }

    return NextResponse.json({
      success: true,
      analytics,
      summary,
    })
  } catch (error) {
    console.error("[v0] Error fetching analytics:", error)
    return NextResponse.json(
      {
        success: false,
        error: error instanceof Error ? error.message : "Unknown error occurred",
        analytics: [],
        summary: null,
      },
      { status: 500 },
    )
  }
}
