import { type NextRequest, NextResponse } from "next/server"
import typesenseClient, { addAd, createAdsCollection } from "@/lib/typesense"

export async function GET(request: NextRequest) {
  try {
    console.log("[ADMIN API] GET request received")

    if (!process.env.TYPESENSE_API_KEY || !process.env.TYPESENSE_HOST) {
      console.log("[ADMIN API] Typesense not configured")
      return NextResponse.json({
        success: false,
        error: "Typesense is not configured",
        ads: [],
      })
    }

    const { searchParams } = new URL(request.url)
    const searchQuery = searchParams.get("search") || ""

    console.log(`[ADMIN API] Fetching ads for admin, search: "${searchQuery}"`)
    console.log(`[ADMIN API] Environment check:`, {
      hasApiKey: !!process.env.TYPESENSE_API_KEY,
      hasHost: !!process.env.TYPESENSE_HOST,
      host: process.env.TYPESENSE_HOST,
      port: process.env.TYPESENSE_PORT,
    })

    // Ensure collection exists
    try {
      await createAdsCollection()
      console.log("[ADMIN API] Collection creation/check completed")
    } catch (createError) {
      console.log("[ADMIN API] Collection creation result:", createError.message)
    }

    try {
      const collectionInfo = await typesenseClient.collections("ads_results").retrieve()
      console.log("[ADMIN API] Collection info:", collectionInfo)
    } catch (collectionError) {
      console.error("[ADMIN API] Failed to retrieve collection info:", collectionError)
    }

    // Get all ads directly from Typesense - NO FILTERING for admin
    let ads = []
    try {
      const searchParams: any = {
        q: searchQuery && searchQuery.trim() ? searchQuery : "*",
        per_page: 250, // Get lots of results for admin
        sort_by: "active:desc",
      }

      // If there's a search query, add query fields
      if (searchQuery && searchQuery.trim()) {
        searchParams.query_by = "title,description,keywords,company,targetUrl"
        searchParams.query_by_weights = "3,2,4,1,1"
      }

      console.log(`[ADMIN API] Search params:`, searchParams)

      const searchResults = await typesenseClient.collections("ads_results").documents().search(searchParams)

      console.log(`[ADMIN API] Raw search response:`, {
        found: searchResults.found,
        hitsLength: searchResults.hits?.length,
        searchTime: searchResults.search_time_ms,
      })

      if (searchResults.hits && searchResults.hits.length > 0) {
        console.log(`[ADMIN API] First ad sample:`, searchResults.hits[0].document)
      }

      ads =
        searchResults.hits?.map((hit: any) => {
          const doc = hit.document
          return {
            id: doc.id,
            title: doc.title || "",
            description: doc.description || "",
            imageUrl: doc.imageUrl || "",
            url: doc.url || doc.targetUrl || "",
            targetUrl: doc.targetUrl || "",
            company: doc.company || "",
            adType: doc.adType || "search",
            keywords: Array.isArray(doc.keywords) ? doc.keywords : [],
            favicon: doc.favicon || "",
            dailyLimit: Number.parseInt(doc.dailyLimit) || 100,
            monthlyLimit: Number.parseInt(doc.monthlyLimit) || 1000,
            dailyShown: Number.parseInt(doc.dailyShown) || 0,
            monthlyShown: Number.parseInt(doc.monthlyShown) || 0,
            lastShownDate: doc.lastShownDate || "",
            active: doc.active !== false,
            createdAt: doc.createdAt || new Date().toISOString(),
          }
        }) || []
    } catch (searchError) {
      console.error("[ADMIN API] Error searching ads:", searchError)
      console.error("[ADMIN API] Search error details:", {
        message: searchError.message,
        httpStatus: searchError.httpStatus,
        stack: searchError.stack,
      })

      // Check if it's a collection not found error
      if (searchError.httpStatus === 404) {
        console.log("[ADMIN API] Collection not found, returning empty result")
        return NextResponse.json({
          success: true,
          ads: [],
          total: 0,
          message: "No ads collection found. Create your first ad to get started.",
        })
      }

      throw searchError
    }

    console.log(`[ADMIN API] Returning ${ads.length} ads to admin`)
    console.log(`[ADMIN API] Sample ads:`, ads.slice(0, 2))

    return NextResponse.json({
      success: true,
      ads: ads,
      total: ads.length,
    })
  } catch (error) {
    console.error("[ADMIN API] Error serving ads:", error)
    return NextResponse.json(
      {
        success: false,
        error: error instanceof Error ? error.message : "Unknown error occurred",
        ads: [],
        debug: {
          hasApiKey: !!process.env.TYPESENSE_API_KEY,
          hasHost: !!process.env.TYPESENSE_HOST,
          errorType: error.constructor.name,
          errorMessage: error instanceof Error ? error.message : "Unknown error",
        },
      },
      { status: 500 },
    )
  }
}

export async function POST(request: NextRequest) {
  try {
    console.log("[ADMIN API] POST request received")

    if (!process.env.TYPESENSE_API_KEY || !process.env.TYPESENSE_HOST) {
      return NextResponse.json(
        {
          success: false,
          error: "Typesense is not configured",
        },
        { status: 500 },
      )
    }

    const body = await request.json()
    console.log("[ADMIN API] Request body:", body)

    const { title, description, imageUrl, targetUrl, company, keywords, dailyLimit, monthlyLimit, active = true } = body

    // Validation
    if (!title || !title.trim()) {
      return NextResponse.json(
        {
          success: false,
          error: "Title is required",
        },
        { status: 400 },
      )
    }

    if (!description || !description.trim()) {
      return NextResponse.json(
        {
          success: false,
          error: "Description is required",
        },
        { status: 400 },
      )
    }

    if (!targetUrl || !targetUrl.trim()) {
      return NextResponse.json(
        {
          success: false,
          error: "Target URL is required",
        },
        { status: 400 },
      )
    }

    // Validate URL format
    try {
      new URL(targetUrl)
    } catch {
      return NextResponse.json(
        {
          success: false,
          error: "Target URL must be a valid URL",
        },
        { status: 400 },
      )
    }

    // Prepare ad data
    const newAd = {
      title: title.trim(),
      description: description.trim(),
      imageUrl: imageUrl ? imageUrl.trim() : "",
      url: targetUrl.trim(),
      targetUrl: targetUrl.trim(),
      company: company ? company.trim() : "",
      adType: "search",
      keywords: Array.isArray(keywords)
        ? keywords.filter((k) => k.trim())
        : keywords
          ? keywords
              .split(",")
              .map((k: string) => k.trim())
              .filter((k) => k)
          : [],
      dailyLimit: Number.parseInt(dailyLimit) || 100,
      monthlyLimit: Number.parseInt(monthlyLimit) || 1000,
      active: Boolean(active),
    }

    console.log("[ADMIN API] Creating ad:", newAd)

    const result = await addAd(newAd)

    console.log("[ADMIN API] Ad created successfully:", result.id)

    return NextResponse.json({
      success: true,
      ad: result,
      message: "Ad created successfully",
    })
  } catch (error) {
    console.error("[ADMIN API] Error creating ad:", error)
    return NextResponse.json(
      {
        success: false,
        error: error instanceof Error ? error.message : "Failed to create ad",
      },
      { status: 500 },
    )
  }
}
