import { type NextRequest, NextResponse } from "next/server"
import typesenseClient from "@/lib/typesense"

export async function POST(request: NextRequest) {
  try {
    if (!process.env.TYPESENSE_API_KEY || !process.env.TYPESENSE_HOST) {
      console.log("[v0] Typesense not configured - missing environment variables")
      return NextResponse.json(
        {
          success: false,
          error:
            "Typesense is not configured. Please add TYPESENSE_API_KEY and TYPESENSE_HOST environment variables in Project Settings.",
        },
        { status: 400 },
      )
    }

    const data = await request.json()
    console.log("[v0] Adding web result with data:", { title: data.title, url: data.url })

    // Validate required fields
    const { title, description, url, safesearch = true, maindomain = false } = data

    if (!title || !description || !url) {
      return NextResponse.json({ success: false, error: "Title, description, and URL are required" }, { status: 400 })
    }

    // Create document with auto-generated ID if not provided
    const document = {
      id: data.id || `web_${Date.now()}_${Math.random().toString(36).substr(2, 9)}`,
      title,
      description,
      url,
      date: data.date || new Date().toISOString().split("T")[0],
      keywords: data.keywords || [],
      safesearch: Boolean(safesearch),
      maindomain: Boolean(maindomain),
      favicon: data.favicon || `https://www.google.com/s2/favicons?domain=${new URL(url).hostname}&sz=32`,
    }

    // Ensure web_results collection exists
    try {
      console.log("[v0] Checking if web_results collection exists...")
      await typesenseClient.collections("web_results").retrieve()
      console.log("[v0] Collection exists, proceeding with document creation")
    } catch (e) {
      console.log("[v0] Collection doesn't exist, creating it...")
      // Create collection if it doesn't exist
      const webSchema = {
        name: "web_results",
        fields: [
          { name: "title", type: "string" },
          { name: "description", type: "string" },
          { name: "url", type: "string" },
          { name: "date", type: "string", optional: true },
          { name: "keywords", type: "string[]", optional: true },
          { name: "safesearch", type: "bool" },
          { name: "maindomain", type: "bool" },
          { name: "favicon", type: "string", optional: true },
        ],
      }

      try {
        await typesenseClient.collections().create(webSchema)
        console.log("[v0] Collection created successfully")
      } catch (createError) {
        console.error("[v0] Failed to create collection:", createError)
        return NextResponse.json(
          {
            success: false,
            error: `Failed to create collection: ${createError instanceof Error ? createError.message : "Unknown error"}`,
          },
          { status: 500 },
        )
      }
    }

    // Insert document
    console.log("[v0] Inserting document into collection...")
    const result = await typesenseClient.collections("web_results").documents().create(document)
    console.log("[v0] Document inserted successfully:", result.id)

    return NextResponse.json({
      success: true,
      message: "Web result added successfully",
      document: result,
    })
  } catch (error) {
    console.error("[v0] Add web result error:", error)

    if (error instanceof Error) {
      if (error.message.includes("Request failed with HTTP code 400")) {
        return NextResponse.json(
          {
            success: false,
            error: "Typesense connection failed. Please check your environment variables in Project Settings.",
          },
          { status: 400 },
        )
      }
      if (error.message.includes("Request failed with HTTP code 401")) {
        return NextResponse.json(
          {
            success: false,
            error: "Invalid Typesense API key. Please check your TYPESENSE_API_KEY in Project Settings.",
          },
          { status: 401 },
        )
      }
    }

    return NextResponse.json(
      {
        success: false,
        error: error instanceof Error ? error.message : "Unknown error occurred while adding web result",
      },
      { status: 500 },
    )
  }
}
