import { type NextRequest, NextResponse } from "next/server"
import typesenseClient from "@/lib/typesense"

export async function POST(request: NextRequest) {
  try {
    const data = await request.json()

    // Validate required fields
    const { title, url, imgurl, safesearch = true } = data

    if (!title || !url || !imgurl) {
      return NextResponse.json({ success: false, error: "Title, URL, and image URL are required" }, { status: 400 })
    }

    // Create document with auto-generated ID if not provided
    const document = {
      id: data.id || `img_${Date.now()}_${Math.random().toString(36).substr(2, 9)}`,
      title,
      url,
      imgurl,
      keywords: data.keywords || [],
      date: data.date || new Date().toISOString().split("T")[0],
      safesearch: Boolean(safesearch),
      favicon: data.favicon || `https://www.google.com/s2/favicons?domain=${new URL(url).hostname}&sz=32`,
    }

    // Ensure image_results collection exists
    try {
      await typesenseClient.collections("image_results").retrieve()
    } catch (e) {
      // Create collection if it doesn't exist
      const imageSchema = {
        name: "image_results",
        fields: [
          { name: "title", type: "string" },
          { name: "url", type: "string" },
          { name: "imgurl", type: "string" },
          { name: "keywords", type: "string[]", optional: true },
          { name: "date", type: "string", optional: true },
          { name: "safesearch", type: "bool" },
          { name: "favicon", type: "string", optional: true },
        ],
      }
      await typesenseClient.collections().create(imageSchema)
    }

    // Insert document
    const result = await typesenseClient.collections("image_results").documents().create(document)

    return NextResponse.json({
      success: true,
      message: "Image result added successfully",
      document: result,
    })
  } catch (error) {
    console.error("Add image result error:", error)
    return NextResponse.json(
      {
        success: false,
        error: error instanceof Error ? error.message : "Unknown error",
      },
      { status: 500 },
    )
  }
}
