import { type NextRequest, NextResponse } from "next/server"
import { addAd } from "@/lib/typesense"

export async function POST(request: NextRequest) {
  try {
    if (!process.env.TYPESENSE_API_KEY || !process.env.TYPESENSE_HOST) {
      return NextResponse.json(
        {
          success: false,
          error: "Typesense is not configured. Please add environment variables in Project Settings.",
        },
        { status: 400 },
      )
    }

    const data = await request.json()

    // Validate required fields
    const { title, description, targetUrl } = data

    if (!title || !description || !targetUrl) {
      return NextResponse.json(
        { success: false, error: "Title, description, and target URL are required" },
        { status: 400 },
      )
    }

    // Create ad document
    const adData = {
      title,
      description,
      imageUrl: data.imageUrl || "",
      targetUrl,
      keywords: data.keywords || [],
      dailyLimit: data.dailyLimit || 100,
      monthlyLimit: data.monthlyLimit || 3000,
      active: data.active !== undefined ? data.active : true,
    }

    const result = await addAd(adData)

    return NextResponse.json({
      success: true,
      message: "Ad added successfully",
      document: result,
    })
  } catch (error) {
    console.error("[v0] Add ad error:", error)

    if (error instanceof Error) {
      if (error.message.includes("Request failed with HTTP code 400")) {
        return NextResponse.json(
          {
            success: false,
            error: "Typesense connection failed. Please check your environment variables in Project Settings.",
          },
          { status: 400 },
        )
      }
      if (error.message.includes("Request failed with HTTP code 401")) {
        return NextResponse.json(
          {
            success: false,
            error: "Invalid Typesense API key. Please check your TYPESENSE_API_KEY in Project Settings.",
          },
          { status: 401 },
        )
      }
    }

    return NextResponse.json(
      {
        success: false,
        error: error instanceof Error ? error.message : "Unknown error occurred while adding ad",
      },
      { status: 500 },
    )
  }
}
