"use client"

import { useState, useEffect } from "react"
import { Button } from "@/components/ui/button"
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card"
import { Input } from "@/components/ui/input"
import { Label } from "@/components/ui/label"
import { Badge } from "@/components/ui/badge"
import { 
  ArrowLeft, Wallet, ArrowUpRight, ArrowDownRight, 
  CreditCard, History, Home, Settings2, Plus, 
  Loader2, Zap 
} from "lucide-react"
import { useAuth } from "@/components/auth-provider"
import { useRouter } from "next/navigation"
import { Logo } from "@/components/logo"

interface Transaction {
  id: string
  type: "credit" | "debit"
  amount: number
  description: string
  adId?: string
  razorpayPaymentId?: string
  createdAt: string
}

declare global {
  interface Window {
    Razorpay: any
  }
}

export default function WalletPage() {
  const { user } = useAuth()
  const router = useRouter()
  const [loading, setLoading] = useState(false)
  const [amount, setAmount] = useState("")
  const [transactions, setTransactions] = useState<Transaction[]>([])
  const [loadingTransactions, setLoadingTransactions] = useState(true)

  useEffect(() => {
    if (!user) {
      router.push("/ads-center/auth")
      return
    }

    const script = document.createElement("script")
    script.src = "https://checkout.razorpay.com/v1/checkout.js"
    script.async = true
    document.body.appendChild(script)
    fetchTransactions()
    return () => { if (document.body.contains(script)) document.body.removeChild(script) }
  }, [user, router])

  const fetchTransactions = async () => {
    if (!user) return
    try {
      const response = await fetch(`/api/wallet/transactions?userId=${user.id}`)
      if (response.ok) {
        const data = await response.json()
        setTransactions(data.transactions || [])
      }
    } catch (error) {
      console.error("Error fetching transactions:", error)
    } finally {
      setLoadingTransactions(false)
    }
  }

  const handleAddFunds = async () => {
    if (!user || !amount || Number.parseFloat(amount) < 1) return
    setLoading(true)
    try {
      const orderResponse = await fetch("/api/razorpay/create-order", {
        method: "POST",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify({
          amount: Number.parseFloat(amount) * 100,
          currency: "INR",
          userId: user.id,
        }),
      })
      if (!orderResponse.ok) throw new Error("Failed to create order")
      const orderData = await orderResponse.json()

      const options = {
        key: orderData.razorpayKeyId,
        amount: orderData.amount,
        currency: orderData.currency,
        name: "IndusSearch Ads",
        description: "Wallet Top-up",
        order_id: orderData.id,
        handler: async (response: any) => {
          const addFundsResponse = await fetch("/api/wallet/add-funds", {
            method: "POST",
            headers: { "Content-Type": "application/json" },
            body: JSON.stringify({
              userId: user.id,
              amount: Number.parseFloat(amount),
              razorpayPaymentId: response.razorpay_payment_id,
              razorpayOrderId: response.razorpay_order_id,
              razorpaySignature: response.razorpay_signature,
            }),
          })
          if (addFundsResponse.ok) window.location.reload()
        },
        prefill: { name: user.username, email: user.email },
        theme: { color: "oklch(0.985 0 0)" },
      }
      const razorpay = new window.Razorpay(options)
      razorpay.open()
    } catch (error) {
      console.error(error)
    } finally {
      setLoading(false)
    }
  }

  if (!user) return null

  return (
    <div className="min-h-screen bg-background text-foreground selection:bg-primary selection:text-primary-foreground">
      {/* Premium Header */}
      <header className="sticky top-0 z-50 border-b border-border/40 bg-background/80 backdrop-blur-md">
        <div className="container mx-auto px-6 h-20 flex justify-between items-center">
          <div className="flex items-center gap-6">
            <Logo size="sm" />
            <div className="h-6 w-[1px] bg-border/40 hidden md:block" />
            <nav className="hidden md:flex items-center gap-2">
              <Button variant="ghost" size="sm" onClick={() => router.push("/")} className="text-[10px] font-bold uppercase tracking-widest px-3">
                <Home className="w-3.5 h-3.5 mr-2" /> Home
              </Button>
              <Button variant="ghost" size="sm" onClick={() => router.push("/settings")} className="text-[10px] font-bold uppercase tracking-widest px-3">
                <Settings2 className="w-3.5 h-3.5 mr-2" /> Settings
              </Button>
            </nav>
          </div>
          <Button variant="ghost" size="sm" onClick={() => router.back()} className="text-[10px] font-bold uppercase tracking-widest border border-border/40 rounded-full px-4">
            <ArrowLeft className="w-3.5 h-3.5 mr-2" /> Back
          </Button>
        </div>
      </header>

      <main className="container mx-auto px-6 py-10">
        <div className="grid grid-cols-1 xl:grid-cols-12 gap-10">
          
          {/* Left: Funding Controls */}
          <div className="xl:col-span-4 space-y-6">
            <section className="space-y-1 mb-8">
              <h2 className="text-3xl font-bold tracking-tight">Wallet</h2>
              <p className="text-muted-foreground italic text-sm font-medium">Power your visibility with instant top-ups.</p>
            </section>

            <Card className="bg-card/30 border-border/50 shadow-none overflow-hidden relative group">
              <div className="absolute top-0 right-0 p-4 opacity-10 group-hover:opacity-20 transition-opacity">
                <Zap className="w-24 h-24" />
              </div>
              <CardHeader className="pb-2">
                <CardTitle className="text-[10px] uppercase tracking-[0.2em] text-muted-foreground flex items-center gap-2">
                   <Wallet className="w-3 h-3" /> Available Balance
                </CardTitle>
              </CardHeader>
              <CardContent>
                <div className="text-4xl font-black tracking-tighter mb-6">
                  ₹{user.walletBalance.toLocaleString('en-IN', { minimumFractionDigits: 2 })}
                </div>
                
                <div className="space-y-4">
                  <div className="space-y-2">
                    <Label className="text-[10px] uppercase tracking-widest font-bold">Custom Amount</Label>
                    <div className="relative">
                      <IndianRupeeIcon className="absolute left-3 top-1/2 -translate-y-1/2 w-3.5 h-3.5 text-muted-foreground" />
                      <Input
                        type="number"
                        value={amount}
                        onChange={(e) => setAmount(e.target.value)}
                        placeholder="0.00"
                        className="pl-9 bg-background/50 border-border/40 h-12 focus:ring-primary/20"
                      />
                    </div>
                  </div>
                  
                  <Button
                    onClick={handleAddFunds}
                    disabled={loading || !amount || parseFloat(amount) < 1}
                    className="w-full h-12 font-bold uppercase tracking-widest text-xs shadow-lg shadow-primary/10 transition-all active:scale-[0.98]"
                  >
                    {loading ? <Loader2 className="animate-spin w-4 h-4" /> : <><Plus className="w-4 h-4 mr-2" /> Add Funds</>}
                  </Button>
                </div>
              </CardContent>
            </Card>

            <Card className="bg-transparent border-border/20 shadow-none">
              <CardHeader className="pb-3">
                <CardTitle className="text-[10px] uppercase tracking-[0.2em]">Quick Presets</CardTitle>
              </CardHeader>
              <CardContent>
                <div className="grid grid-cols-3 gap-2">
                  {[500, 1000, 2000, 5000, 10000, 25000].map((q) => (
                    <Button
                      key={q}
                      variant="outline"
                      size="sm"
                      onClick={() => setAmount(q.toString())}
                      className="text-[10px] font-bold border-border/40 hover:bg-secondary/50"
                    >
                      ₹{q.toLocaleString()}
                    </Button>
                  ))}
                </div>
              </CardContent>
            </Card>
          </div>

          {/* Right: History */}
          <div className="xl:col-span-8 space-y-6">
            <div className="flex items-center justify-between px-2">
              <div className="flex items-center gap-2">
                <History className="w-4 h-4 text-muted-foreground" />
                <h3 className="text-xs font-black uppercase tracking-[0.3em] text-muted-foreground">Ledger Activity</h3>
              </div>
              <Badge variant="outline" className="text-[9px] border-border/40 text-muted-foreground uppercase">{transactions.length} entries</Badge>
            </div>

            <Card className="bg-card/20 border-border/50 backdrop-blur-sm shadow-none">
              <CardContent className="p-0">
                {loadingTransactions ? (
                  <div className="p-20 text-center"><Loader2 className="animate-spin w-6 h-6 mx-auto opacity-20" /></div>
                ) : transactions.length === 0 ? (
                  <div className="p-20 text-center space-y-4">
                    <div className="w-12 h-12 rounded-full bg-secondary/30 flex items-center justify-center mx-auto">
                      <CreditCard className="w-5 h-5 text-muted-foreground" />
                    </div>
                    <p className="text-xs font-bold uppercase tracking-widest text-muted-foreground">No recent activity found</p>
                  </div>
                ) : (
                  <div className="divide-y divide-border/20">
                    {transactions.map((tx) => (
                      <div key={tx.id} className="p-5 flex items-center justify-between hover:bg-secondary/10 transition-colors group">
                        <div className="flex items-center gap-4">
                          <div className={`w-10 h-10 rounded-full flex items-center justify-center border ${
                            tx.type === "credit" ? "bg-green-500/5 border-green-500/20 text-green-500" : "bg-red-500/5 border-red-500/20 text-red-500"
                          }`}>
                            {tx.type === "credit" ? <ArrowDownRight className="w-4 h-4" /> : <ArrowUpRight className="w-4 h-4" />}
                          </div>
                          <div className="space-y-1">
                            <p className="text-sm font-bold tracking-tight">{tx.description}</p>
                            <p className="text-[10px] text-muted-foreground font-mono uppercase">
                              {new Date(tx.createdAt).toLocaleDateString("en-IN", { day: '2-digit', month: 'short', year: 'numeric', hour: '2-digit', minute: '2-digit' })}
                            </p>
                          </div>
                        </div>
                        <div className="text-right space-y-1">
                          <p className={`text-sm font-black ${tx.type === "credit" ? "text-green-500" : "text-foreground"}`}>
                            {tx.type === "credit" ? "+" : "-"} ₹{tx.amount.toLocaleString('en-IN', { minimumFractionDigits: 2 })}
                          </p>
                          <Badge variant="outline" className="text-[8px] h-4 uppercase font-bold tracking-tighter border-border/20">
                            {tx.type}
                          </Badge>
                        </div>
                      </div>
                    ))}
                  </div>
                )}
              </CardContent>
            </Card>
          </div>
        </div>
      </main>
    </div>
  )
}

function IndianRupeeIcon({ className }: { className?: string }) {
  return (
    <svg 
      viewBox="0 0 24 24" 
      fill="none" 
      stroke="currentColor" 
      strokeWidth="2" 
      strokeLinecap="round" 
      strokeLinejoin="round" 
      className={className}
    >
      <path d="M6 3h12" />
      <path d="M6 8h12" />
      <path d="m6 13 8.5 8" />
      <path d="M6 13h3" />
      <path d="M9 13c6.667 0 6.667-10 0-10" />
    </svg>
  )
}