"use client"

import { useState, useEffect } from "react"
import { Button } from "@/components/ui/button"
import { Card, CardContent } from "@/components/ui/card"
import { Badge } from "@/components/ui/badge"
import {
  Plus, TrendingUp, Eye, MousePointer, Wallet,
  BarChart3, LogOut, ExternalLink, Settings2, PauseCircle, PlayCircle, Trash2, Home
} from "lucide-react"
import { useAuth } from "@/components/auth-provider"
import { useRouter } from "next/navigation"
import { Logo, LogoWithText } from "@/components/logo"

export default function AdsCenterPage() {
  const { user, logout, loading: authLoading } = useAuth()
  const router = useRouter()
  const [ads, setAds] = useState<any[]>([])
  const [loading, setLoading] = useState(true)

  useEffect(() => {
    if (authLoading) return
    if (!user) { router.push("/ads-center/auth"); return }
    fetchUserAds()
  }, [user, router, authLoading])

  const fetchUserAds = async () => {
    try {
      const response = await fetch(`/api/user-ads?userId=${user?.id}`)
      if (response.ok) {
        const data = await response.json()
        setAds(data.ads || [])
      }
    } catch (error) { console.error(error) } finally { setLoading(false) }
  }

  // Derived Stats
  const stats = [
    { label: "Active Campaigns", value: ads.filter(a => a.isActive).length, icon: TrendingUp, suffix: `of ${ads.length}` },
    { label: "Total Investment", value: `₹${ads.reduce((s, a) => s + a.totalSpent, 0).toLocaleString()}`, icon: Wallet, suffix: "this month" },
    { label: "Engagement", value: ads.reduce((s, a) => s + a.clicks, 0).toLocaleString(), icon: MousePointer, suffix: "total clicks" },
    { label: "Visibility", value: ads.reduce((s, a) => s + a.impressions, 0).toLocaleString(), icon: Eye, suffix: "total impressions" },
  ]

  if (authLoading) return (
    <div className="min-h-screen bg-background flex items-center justify-center">
      <div className="w-6 h-6 border-2 border-primary border-t-transparent rounded-full animate-spin" />
    </div>
  )

  return (
    <div className="min-h-screen bg-background text-foreground selection:bg-primary selection:text-primary-foreground">
      {/* Premium Header */}
      {/* Premium Header */}
      <header className="sticky top-0 z-50 border-b border-border/40 bg-background/80 backdrop-blur-md">
        <div className="container mx-auto px-6 h-20 flex justify-between items-center">
          <div className="flex items-center gap-6">
            <LogoWithText size="sm" />
            <div className="h-6 w-[1px] bg-border/60 hidden md:block" />

            {/* Navigation Links */}
            <nav className="hidden md:flex items-center gap-2">
              <Button
                variant="ghost"
                size="sm"
                onClick={() => router.push("/")}
                className="text-[10px] font-bold uppercase tracking-widest hover:bg-secondary/50 rounded-lg px-3"
              >
                <Home className="w-3.5 h-3.5 mr-2" />
          Home
        </Button>
              <Button
                variant="ghost"
                size="sm"
                onClick={() => router.push("/settings")}
                className="text-[10px] font-bold uppercase tracking-widest hover:bg-secondary/50 rounded-lg px-3"
              >
                <Settings2 className="w-3.5 h-3.5 mr-2" />
          Settings
        </Button>
            </nav>
          </div>

          <div className="flex items-center gap-4">
            <div className="hidden sm:flex flex-col items-end mr-2">
              <span className="text-[10px] font-bold text-muted-foreground uppercase tracking-widest">Balance</span>
              <span className="text-sm font-mono font-bold text-primary">₹{user?.walletBalance.toFixed(2)}</span>
            </div>

            {/* Action Group */}
            <div className="flex items-center gap-2 border-l border-border/40 pl-4">
            <Button
                variant="outline"
                size="icon"
                onClick={() => router.push("/")}
                className="md:hidden rounded-full border-border/40 bg-secondary/30"
              >
                <Home className="w-4 h-4" />
              </Button>

              <Button
                variant="outline"
                size="icon"
                onClick={() => router.push("/settings")}
                className="md:hidden rounded-full border-border/40 bg-secondary/30"
              >
                <Settings2 className="w-4 h-4" />
              </Button>
              <Button
                variant="outline"
                size="sm"
                onClick={() => router.push("/ads-center/wallet")}
                className="rounded-full border-border/40 bg-secondary/30 hover:bg-secondary/80 transition-all"
              >
                <Plus className="w-3.5 h-3.5 mr-2" /> Top Up
        </Button>
              <Button variant="ghost" size="icon" onClick={logout} className="text-muted-foreground hover:text-white transition-colors">
                <LogOut className="w-4 h-4" />
              </Button>
            </div>
          </div>
        </div>
      </header>

      <main className="container mx-auto px-6 py-12 space-y-12">
        {/* Welcome Section */}
        <section className="flex flex-col md:flex-row md:items-end justify-between gap-6">
          <div className="space-y-2">
            <h2 className="text-4xl font-bold tracking-tight">Overview</h2>
            <p className="text-muted-foreground text-lg italic">Welcome back, {user?.username}.</p>
          </div>
          <Button
            onClick={() => router.push("/ads-center/create")}
            className="h-12 px-8 rounded-xl bg-primary text-primary-foreground font-bold uppercase tracking-widest hover:scale-[1.02] active:scale-[0.98] transition-all"
          >
            Create Campaign
          </Button>
        </section>

        {/* Stats Grid */}
        <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-4 gap-4">
          {stats.map((stat, i) => (
            <Card key={i} className="bg-card/40 border-border/40 shadow-none hover:border-primary/20 transition-colors group">
              <CardContent className="p-6">
                <div className="flex justify-between items-start mb-4">
                  <span className="text-[10px] font-bold uppercase tracking-widest text-muted-foreground group-hover:text-primary transition-colors">
                    {stat.label}
                  </span>
                  <stat.icon className="w-4 h-4 text-muted-foreground/50" />
                </div>
                <div className="text-3xl font-bold font-mono tracking-tighter">{stat.value}</div>
                <div className="text-[10px] uppercase text-muted-foreground/60 mt-1">{stat.suffix}</div>
              </CardContent>
            </Card>
          ))}
        </div>

        {/* Ads Section */}
        <div className="space-y-6">
          <h3 className="text-xs font-black uppercase tracking-[0.3em] text-muted-foreground/50">Your Campaigns</h3>

          {loading ? (
            <div className="flex flex-col items-center justify-center py-20 border border-dashed border-border/40 rounded-3xl">
              <div className="w-8 h-8 border-2 border-primary border-t-transparent rounded-full animate-spin mb-4" />
              <p className="text-xs uppercase tracking-widest text-muted-foreground">Synchronizing Data...</p>
            </div>
          ) : ads.length === 0 ? (
            <div className="flex flex-col items-center justify-center py-24 border border-dashed border-border/40 rounded-3xl bg-secondary/10">
              <BarChart3 className="w-12 h-12 text-muted-foreground/20 mb-6" />
              <h4 className="text-xl font-bold mb-2">No Active Campaigns</h4>
              <p className="text-muted-foreground text-sm mb-8">Reach millions of users by creating your first ad today.</p>
              <Button onClick={() => router.push("/ads-center/create")} variant="secondary" className="rounded-full px-8 border-border">
                Start Now
              </Button>
            </div>
          ) : (
                <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
                  {ads.map((ad) => (
                    <Card key={ad.id} className="group overflow-hidden border-border/40 bg-card/20 backdrop-blur-sm transition-all hover:bg-card/40 hover:border-border">
                      <CardContent className="p-0">
                        <div className="flex flex-col sm:flex-row h-full">
                          {/* Image/Preview */}
                          <div className="sm:w-48 h-48 sm:h-auto relative bg-secondary/50 overflow-hidden">
                            {ad.imageUrl ? (
                              <img src={ad.imageUrl} alt="" className="w-full h-full object-cover grayscale hover:grayscale-0 transition-all duration-500" />
                            ) : (
                                <div className="w-full h-full flex items-center justify-center"><BarChart3 className="w-8 h-8 opacity-10" /></div>
                              )}
                            <Badge className="absolute top-3 left-3 rounded-full uppercase text-[9px] font-black" variant={ad.isActive ? "default" : "secondary"}>
                              {ad.isActive ? "Live" : "Paused"}
                            </Badge>
                          </div>

                          {/* Content */}
                          <div className="flex-1 p-6 flex flex-col justify-between space-y-4">
                            <div className="space-y-2">
                              <div className="flex justify-between items-start">
                                <h4 className="font-bold text-lg leading-tight group-hover:text-primary transition-colors">{ad.title}</h4>
                                <a href={ad.targetUrl} target="_blank" className="p-1 hover:bg-secondary rounded transition-colors">
                                  <ExternalLink className="w-4 h-4 text-muted-foreground" />
                                </a>
                              </div>
                              <p className="text-xs text-muted-foreground line-clamp-2 leading-relaxed">{ad.description}</p>
                            </div>

                            <div className="grid grid-cols-3 gap-2 border-t border-border/20 pt-4">
                              <div className="flex flex-col">
                                <span className="text-[9px] font-bold uppercase text-muted-foreground">Clicks</span>
                                <span className="text-sm font-mono font-bold">{ad.clicks}</span>
                              </div>
                              <div className="flex flex-col">
                                <span className="text-[9px] font-bold uppercase text-muted-foreground">Reach</span>
                                <span className="text-sm font-mono font-bold">{ad.impressions}</span>
                              </div>
                              <div className="flex flex-col">
                                <span className="text-[9px] font-bold uppercase text-muted-foreground">Spend</span>
                                <span className="text-sm font-mono font-bold text-primary">₹{ad.totalSpent.toFixed(0)}</span>
                              </div>
                            </div>

                            <div className="flex gap-2 pt-2">
                              <Button
                                variant="secondary"
                                size="sm"
                                className="flex-1 text-[10px] font-bold uppercase rounded-lg border-border/40"
                                onClick={() => router.push(`/ads-center/edit/${ad.id}`)}
                              >
                                <Settings2 className="w-3 h-3 mr-2" /> Edit
                          </Button>
                              <Button
                                variant="secondary"
                                size="sm"
                                className="flex-1 text-[10px] font-bold uppercase rounded-lg border-border/40"
                                onClick={() => {/* Toggle Logic */ }}
                              >
                                {ad.isActive ? <PauseCircle className="w-3 h-3 mr-2 text-primary" /> : <PlayCircle className="w-3 h-3 mr-2" />}
                                {ad.isActive ? "Pause" : "Resume"}
                              </Button>
                              <Button variant="ghost" size="sm" className="px-2 hover:bg-destructive/10 hover:text-white">
                                <Trash2 className="w-4 h-4" />
                              </Button>
                            </div>
                          </div>
                        </div>
                      </CardContent>
                    </Card>
                  ))}
                </div>
              )}
        </div>
      </main>

      {/* Subtle Footer */}
      <footer className="container mx-auto px-6 py-12 border-t border-border/20 text-center">
        <p className="text-[10px] uppercase tracking-[0.5em] text-muted-foreground/40">Secure Ad Infrastructure provided by IndzsSearch</p>
      </footer>
    </div>
  )
}