"use client"

import type React from "react"
import { useState, useEffect } from "react"
import { Button } from "@/components/ui/button"
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card"
import { Input } from "@/components/ui/input"
import { Label } from "@/components/ui/label"
import { Textarea } from "@/components/ui/textarea"
import { Badge } from "@/components/ui/badge"
import { 
  ArrowLeft, Eye, Plus, X, Globe, Type, Image as ImageIcon, 
  Target, Hash, IndianRupee, Sparkles, Loader2, AlertCircle,
  Home, Settings2
} from "lucide-react"
import { useAuth } from "@/components/auth-provider"
import { useRouter } from "next/navigation"
import { Logo } from "@/components/logo"

export default function CreateAdPage() {
  const { user } = useAuth()
  const router = useRouter()
  
  // States
  const [loading, setLoading] = useState(false)
  const [keywordInput, setKeywordInput] = useState("")
  const [errors, setErrors] = useState<Record<string, string>>({})
  const [adData, setAdData] = useState({
    title: "",
    description: "",
    imageUrl: "",
    targetUrl: "",
    keywords: [] as string[],
    dailyLimit: 100,
    monthlyLimit: 1000,
    costPerClick: 0.5,
    costPerImpression: 0.1,
  })

  useEffect(() => {
    if (!user) router.push("/ads-center/auth")
  }, [user, router])

  // Helpers
  const getHostname = (url: string) => {
    try {
      if (!url) return "example.com"
      return new URL(url).hostname
    } catch { return "example.com" }
  }

  const addKeyword = () => {
    if (keywordInput.trim() && !adData.keywords.includes(keywordInput.trim())) {
      console.log(keywordInput.trim().split(','))
      setAdData({ ...adData, keywords: [...adData.keywords, ...keywordInput.trim().split(',')] })
      setKeywordInput("")
      if (errors.keywords) setErrors(prev => { const { keywords, ...rest } = prev; return rest })
    }
  }

  

  const removeKeyword = (keyword: string) => {
    console.log("removed clicked")
    setAdData({ ...adData, keywords: adData.keywords.filter((k) => k !== keyword) })
  }

  // Validation Logic
  const validateForm = () => {
    const newErrors: Record<string, string> = {}
    if (!adData.title.trim()) newErrors.title = "Headline required"
    if (adData.title.length > 60) newErrors.title = "Max 60 chars"
    if (!adData.description.trim()) newErrors.description = "Description required"
    if (!adData.targetUrl.trim()) {
      newErrors.targetUrl = "URL required"
    } else if (!/^https?:\/\/.*/.test(adData.targetUrl)) {
      newErrors.targetUrl = "Must start with http/https"
    }
    if (adData.keywords.length === 0) newErrors.keywords = "Add at least 1 keyword"
    if (adData.dailyLimit <= 0) newErrors.dailyLimit = "Must be > 0"
    if (adData.costPerClick < 0.01) newErrors.costPerClick = "Min ₹0.01"

    setErrors(newErrors)
    return Object.keys(newErrors).length === 0
  }

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault()
    if (!user || !validateForm()) return

    setLoading(true)
    try {
      const response = await fetch("/api/user-ads", {
        method: "POST",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify({ ...adData, userId: user.id }),
      })

      if (response.ok) router.push("/ads-center")
    } catch (error) {
      console.error("Error creating ad:", error)
    } finally {
      setLoading(false)
    }
  }

  if (!user) return null

  return (
    <div className="min-h-screen bg-background text-foreground selection:bg-primary selection:text-primary-foreground">
      {/* Premium Header */}
      <header className="sticky top-0 z-50 border-b border-border/40 bg-background/80 backdrop-blur-md">
        <div className="container mx-auto px-6 h-20 flex justify-between items-center">
          <div className="flex items-center gap-6">
            <Logo size="sm" />
            <div className="h-6 w-[1px] bg-border/40 hidden md:block" />
            <nav className="hidden md:flex items-center gap-2">
              <Button variant="ghost" size="sm" onClick={() => router.push("/")} className="text-[10px] font-bold uppercase tracking-widest px-3">
                <Home className="w-3.5 h-3.5 mr-2" /> Home
              </Button>
              <Button variant="ghost" size="sm" onClick={() => router.push("/settings")} className="text-[10px] font-bold uppercase tracking-widest px-3">
                <Settings2 className="w-3.5 h-3.5 mr-2" /> Settings
              </Button>
            </nav>
          </div>
          <Button variant="ghost" size="sm" onClick={() => router.back()} className="text-[10px] font-bold uppercase tracking-widest border border-border/40 rounded-full px-4">
            <ArrowLeft className="w-3.5 h-3.5 mr-2" /> Cancel
          </Button>
        </div>
      </header>

      <main className="container mx-auto px-6 py-10">
        <div className="grid grid-cols-1 xl:grid-cols-12 gap-10">
          
          {/* Form Side */}
          <div className="xl:col-span-7 space-y-8">
            <header className="space-y-1">
              <h2 className="text-3xl font-bold tracking-tight">Campaign Designer</h2>
              <p className="text-muted-foreground italic">Target users with high-intent search ads.</p>
            </header>

            <form onSubmit={handleSubmit} className="space-y-6">
              <Card className="bg-card/30 border-border/50 shadow-none">
                <CardContent className="p-6 space-y-6">
                  {/* Title & URL */}
                  <div className="grid md:grid-cols-2 gap-6">
                    <div className="space-y-2">
                      <div className="flex justify-between">
                        <Label className={`text-[10px] uppercase tracking-widest font-bold flex items-center gap-2 ${errors.title ? 'text-destructive' : 'text-muted-foreground'}`}>
                          <Type className="w-3 h-3" /> Ad Headline
                        </Label>
                        {errors.title && <span className="text-[10px] text-destructive font-bold uppercase">{errors.title}</span>}
                      </div>
                      <Input
                        value={adData.title}
                        onChange={(e) => { setAdData({ ...adData, title: e.target.value }); if (errors.title) setErrors(p => ({...p, title: ""})) }}
                        placeholder="e.g. Best AI Search Tool"
                        className={`bg-background/50 transition-all ${errors.title ? 'border-destructive/50 ring-destructive/10' : ''}`}
                      />
                    </div>
                    <div className="space-y-2">
                      <div className="flex justify-between">
                        <Label className={`text-[10px] uppercase tracking-widest font-bold flex items-center gap-2 ${errors.targetUrl ? 'text-destructive' : 'text-muted-foreground'}`}>
                          <Globe className="w-3 h-3" /> Destination URL
                        </Label>
                        {errors.targetUrl && <span className="text-[10px] text-destructive font-bold uppercase">{errors.targetUrl}</span>}
                      </div>
                      <Input
                        type="url"
                        value={adData.targetUrl}
                        onChange={(e) => { setAdData({ ...adData, targetUrl: e.target.value }); if (errors.targetUrl) setErrors(p => ({...p, targetUrl: ""})) }}
                        placeholder="https://example.com"
                        className={`bg-background/50 ${errors.targetUrl ? 'border-destructive/50' : ''}`}
                      />
                    </div>
                  </div>

                  {/* Description */}
                  <div className="space-y-2">
                    <div className="flex justify-between">
                      <Label className={`text-[10px] uppercase tracking-widest font-bold flex items-center gap-2 ${errors.description ? 'text-destructive' : 'text-muted-foreground'}`}>
                        <Sparkles className="w-3 h-3" /> Creative Copy
                      </Label>
                      {errors.description && <span className="text-[10px] text-destructive font-bold uppercase">{errors.description}</span>}
                    </div>
                    <Textarea
                      value={adData.description}
                      onChange={(e) => { setAdData({ ...adData, description: e.target.value }); if (errors.description) setErrors(p => ({...p, description: ""})) }}
                      placeholder="Enter the ad text users will see..."
                      rows={3}
                      className={`bg-background/50 resize-none ${errors.description ? 'border-destructive/50' : ''}`}
                    />
                  </div>

                  {/* Keywords */}
                  <div className="space-y-2">
                    <div className="flex justify-between">
                      <Label className={`text-[10px] uppercase tracking-widest font-bold flex items-center gap-2 ${errors.keywords ? 'text-destructive' : 'text-muted-foreground'}`}>
                        <Hash className="w-3 h-3" /> Keywords (Comma Separated)
                      </Label>
                      {errors.keywords && <span className="text-[10px] text-destructive font-bold uppercase">{errors.keywords}</span>}
                    </div>
                    <div className="flex gap-2">
                      <Input
                        value={keywordInput}
                        onChange={(e) => setKeywordInput(e.target.value)}
                        placeholder="Add targeting term..."
                        onKeyDown={(e) => e.key === "Enter" && (e.preventDefault(), addKeyword())}
                        className="bg-background/50"
                      />
                      <Button type="button" onClick={addKeyword} variant="secondary"><Plus className="w-4 h-4" /></Button>
                    </div>
                    <div className="flex flex-wrap gap-2 mt-3">
                      {adData.keywords.map((kw) => (
                        <Badge key={kw} variant="secondary" className="bg-primary/5 text-primary border-primary/10 px-2 py-1 flex items-center gap-1 group" onClick={() => removeKeyword(kw)} >
                          {kw} <X className="w-6 h-6 cursor-pointer opacity-50 group-hover:opacity-100" 
                            style={{
                              height:'20px', width:'20px'
                            }}
                            />
                        </Badge>
                      ))}
                    </div>
                  </div>
                </CardContent>
              </Card>

              {/* Financials */}
              <Card className="bg-card/30 border-border/50 shadow-none">
                <CardContent className="p-6 grid md:grid-cols-2 gap-6">
                  <div className="space-y-4">
                    <div className="space-y-2">
                      <Label className="text-[10px] uppercase tracking-widest font-bold text-muted-foreground">Daily Limit (₹)</Label>
                      <Input type="number" value={adData.dailyLimit} onChange={(e) => setAdData({...adData, dailyLimit: +e.target.value})} className="bg-background/50" />
                    </div>
                    <div className="space-y-2">
                      <Label className="text-[10px] uppercase tracking-widest font-bold text-primary">Cost per Click (₹)</Label>
                      <Input type="number" step="0.01" value={adData.costPerClick} onChange={(e) => setAdData({...adData, costPerClick: +e.target.value})} className="bg-background/50 border-primary/20" />
                    </div>
                  </div>
                  <div className="space-y-4">
                    <div className="space-y-2">
                      <Label className="text-[10px] uppercase tracking-widest font-bold text-muted-foreground text-right">Monthly Limit (₹)</Label>
                      <Input type="number" value={adData.monthlyLimit} onChange={(e) => setAdData({...adData, monthlyLimit: +e.target.value})} className="bg-background/50 text-right" />
                    </div>
                    <div className="space-y-2">
                      <Label className="text-[10px] uppercase tracking-widest font-bold text-muted-foreground text-right">Cost per Impression (₹)</Label>
                      <Input type="number" step="0.01" value={adData.costPerImpression} onChange={(e) => setAdData({...adData, costPerImpression: +e.target.value})} className="bg-background/50 text-right" />
                    </div>
                  </div>
                </CardContent>
              </Card>

              <div className="space-y-4 pt-4">
                {Object.keys(errors).length > 0 && (
                  <div className="p-3 rounded-lg bg-destructive/10 border border-destructive/20 flex items-center gap-3">
                    <AlertCircle className="w-4 h-4 text-destructive" />
                    <span className="text-[10px] font-bold text-destructive uppercase tracking-widest">Correct highlighting issues to deploy</span>
                  </div>
                )}
                <Button 
                  type="submit" 
                  disabled={loading}
                  className="w-full h-14 text-sm font-bold uppercase tracking-[0.2em] shadow-lg shadow-primary/10 transition-all active:scale-[0.98]"
                >
                  {loading ? <Loader2 className="w-5 h-5 animate-spin" /> : "Deploy Campaign"}
                </Button>
              </div>
            </form>
          </div>

          {/* Preview Side */}
          <div className="xl:col-span-5 relative">
            <div className="sticky top-32 space-y-6">
              <h3 className="text-xs font-black uppercase tracking-[0.3em] text-muted-foreground px-2">Live Search Preview</h3>
              
              <Card className="overflow-hidden border-border/40 bg-card/20 backdrop-blur-sm relative group">
                <div className="p-6 space-y-4">
                  <div className="flex items-center gap-2">
                    <div className="bg-primary/10 text-primary text-[10px] font-black px-1.5 py-0.5 rounded border border-primary/20">AD</div>
                    <span className="text-xs text-muted-foreground font-medium flex items-center gap-1">
                      <Globe className="w-3 h-3" /> {getHostname(adData.targetUrl)}
                    </span>
                  </div>

                  <div className="space-y-2">
                    <h4 className="text-xl font-bold text-primary hover:underline cursor-pointer leading-tight decoration-primary/30">
                      {adData.title || "Your Campaign Headline"}
                    </h4>
                    <p className="text-sm text-muted-foreground leading-relaxed line-clamp-3 italic">
                      {adData.description || "Start typing your description to see it appear here in the search results..."}
                    </p>
                  </div>

                  <div className="flex items-center justify-between pt-4 border-t border-border/10">
                    <div className="flex gap-1.5">
                      {adData.keywords.slice(0, 3).map(k => (
                        <span key={k} className="text-[9px] font-bold uppercase text-muted-foreground/50">#{k}</span>
                      ))}
                    </div>
                    <span className="text-[10px] font-bold text-primary tracking-widest uppercase">Sponsored</span>
                  </div>
                </div>
                {/* Visual Polish */}
                <div className="absolute inset-0 pointer-events-none bg-gradient-to-br from-primary/5 to-transparent opacity-40" />
              </Card>

              <div className="p-6 rounded-3xl bg-secondary/20 border border-border/40">
                <div className="flex justify-between items-center mb-4">
                  <span className="text-[10px] font-bold uppercase tracking-widest text-muted-foreground">Est. Daily Reach</span>
                  <span className="text-sm font-mono font-bold text-primary">
                    {Math.floor(adData.dailyLimit / (adData.costPerImpression || 0.1)).toLocaleString()}
                  </span>
                </div>
                <div className="h-[1px] bg-border/20 w-full mb-4" />
                <p className="text-[10px] text-muted-foreground uppercase tracking-widest leading-loose">
                  Credits will be deducted from your wallet based on real-time engagement.
                </p>
              </div>
            </div>
          </div>

        </div>
      </main>
    </div>
  )
}