"use client"

import type React from "react"

import { useState } from "react"
import { Button } from "@/components/ui/button"
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card"
import { Input } from "@/components/ui/input"
import { Label } from "@/components/ui/label"
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs"
import { Alert, AlertDescription } from "@/components/ui/alert"
import { Logo } from "@/components/logo"
import { useAuth } from "@/components/auth-provider"
import { useRouter } from "next/navigation"
import { Eye, EyeOff, CheckCircle, XCircle, Mail, AlertCircle } from "lucide-react"
import { ArrowLeft, ShieldCheck, Lock, User } from "lucide-react"
export default function AuthPage() {

  const { login, register } = useAuth()
  const router = useRouter()
  const [loading, setLoading] = useState(false)
  const [error, setError] = useState("")
  const [showPassword, setShowPassword] = useState(false)
  const [usernameAvailable, setUsernameAvailable] = useState<boolean | null>(null)
  const [checkingUsername, setCheckingUsername] = useState(false)
  const [showVerificationMessage, setShowVerificationMessage] = useState(false)
  const [verificationEmail, setVerificationEmail] = useState("")
  const [resendingVerification, setResendingVerification] = useState(false)
  const [showForgotPassword, setShowForgotPassword] = useState(false)
  const [forgotPasswordEmail, setForgotPasswordEmail] = useState("")
  const [forgotPasswordLoading, setForgotPasswordLoading] = useState(false)
  const [forgotPasswordSent, setForgotPasswordSent] = useState(false)

  router.push("/settings")
  // Login form state
  const [loginData, setLoginData] = useState({
    username: "",
    password: "",
  })

  // Register form state
  const [registerData, setRegisterData] = useState({
    username: "",
    email: "",
    password: "",
    confirmPassword: "",
  })

  const checkUsernameAvailability = async (username: string) => {
    if (username.length < 3) {
      setUsernameAvailable(null)
      return
    }

    setCheckingUsername(true)
    try {
      const response = await fetch(`/api/auth/check-username?username=${encodeURIComponent(username)}`)
      const data = await response.json()
      setUsernameAvailable(data.available)
    } catch (error) {
      console.error("Error checking username:", error)
      setUsernameAvailable(null)
    } finally {
      setCheckingUsername(false)
    }
  }

  const handleLogin = async (e: React.FormEvent) => {
    e.preventDefault()
    setLoading(true)
    setError("")

    const result = await login(loginData.username, loginData.password)
    if (result.success) {
      router.push("/ads-center")
    } else {
      if (result.requiresVerification) {
        setError("Please verify your email address before signing in. Check your inbox for a verification link.")
        // setVerificationEmail(loginData.username)
        // setShowVerificationMessage(true)
      } else {
        setError(result.error || "Invalid username or password")
      }
    }
    setLoading(false)
  }

  const handleRegister = async (e: React.FormEvent) => {
    e.preventDefault()
    setLoading(true)
    setError("")

    if (registerData.password !== registerData.confirmPassword) {
      setError("Passwords do not match")
      setLoading(false)
      return
    }

    if (registerData.password.length < 6) {
      setError("Password must be at least 6 characters long")
      setLoading(false)
      return
    }

    if (!usernameAvailable) {
      setError("Username is not available")
      setLoading(false)
      return
    }

    const result = await register(registerData.username, registerData.email, registerData.password)
    if (result.success) {
      if (result.requiresVerification) {
        setVerificationEmail(registerData.email)
        setShowVerificationMessage(true)
        setError("")
      } else {
        router.push("/ads-center")
      }
    } else {
      setError(result.error || "Registration failed. Please try again.")
    }
    setLoading(false)
  }

  const handleResendVerification = async () => {
    setResendingVerification(true)
    try {
      const response = await fetch("/api/auth/resend-verification", {
        method: "POST",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify({ email: verificationEmail }),
      })

      if (response.ok) {
        setError("")
        // Show success message temporarily
        const originalError = error
        setError("Verification email sent successfully!")
        setTimeout(() => setError(originalError), 3000)
      } else {
        const data = await response.json()
        setError(data.error || "Failed to send verification email")
      }
    } catch (error) {
      setError("Network error occurred")
    } finally {
      setResendingVerification(false)
    }
  }

  const handleForgotPassword = async (e: React.FormEvent) => {
    e.preventDefault()
    setForgotPasswordLoading(true)
    setError("")

    try {
      const response = await fetch("/api/auth/forgot-password", {
        method: "POST",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify({ email: forgotPasswordEmail }),
      })

      const data = await response.json()

      if (response.ok) {
        setForgotPasswordSent(true)
        setError("")
      } else {
        setError(data.error || "Failed to send reset email")
      }
    } catch (error) {
      setError("Network error occurred")
    } finally {
      setForgotPasswordLoading(false)
    }
  }

  // if (showForgotPassword && forgotPasswordSent) {
  //   return (
  //     <div className="min-h-screen bg-background flex items-center justify-center p-4">
  //       <div className="w-full max-w-md">
  //         <div className="text-center mb-8">
  //           <Logo size="md" className="mx-auto mb-4" />
  //           <h1 className="text-2xl font-semibold text-foreground">Check Your Email</h1>
  //           <p className="text-muted-foreground">Password reset link sent</p>
  //         </div>

  //         <Card>
  //           <CardHeader className="text-center">
  //             <div className="mx-auto mb-4 w-12 h-12 rounded-full bg-primary/10 flex items-center justify-center">
  //               <Mail className="w-6 h-6 text-primary" />
  //             </div>
  //             <CardTitle>Reset Link Sent</CardTitle>
  //             <CardDescription>
  //               We've sent a password reset link to <strong>{forgotPasswordEmail}</strong>
  //             </CardDescription>
  //           </CardHeader>
  //           <CardContent className="space-y-4">
  //             <Alert>
  //               <AlertCircle className="h-4 w-4" />
  //               <AlertDescription>
  //                 Please check your email and click the reset link to create a new password. The link will expire in 1
  //                 hour.
  //               </AlertDescription>
  //             </Alert>

  //             <div className="space-y-2">
  //               <Button
  //                 onClick={() => {
  //                   setShowForgotPassword(false)
  //                   setForgotPasswordSent(false)
  //                   setForgotPasswordEmail("")
  //                 }}
  //                 variant="outline"
  //                 className="w-full"
  //               >
  //                 Back to Sign In
  //               </Button>
  //             </div>
  //           </CardContent>
  //         </Card>
  //       </div>
  //     </div>
  //   )
  // }

  // if (showForgotPassword) {
  //   return (
  //     <div className="min-h-screen bg-background flex items-center justify-center p-4">
  //       <div className="w-full max-w-md">
  //         <div className="text-center mb-8">
  //           <Logo size="md" className="mx-auto mb-4" />
  //           <h1 className="text-2xl font-semibold text-foreground">Reset Password</h1>
  //           <p className="text-muted-foreground">Enter your email to receive a reset link</p>
  //         </div>

  //         <Card>
  //           <CardHeader>
  //             <CardTitle>Forgot Password</CardTitle>
  //             <CardDescription>
  //               Enter your email address and we'll send you a link to reset your password
  //             </CardDescription>
  //           </CardHeader>
  //           <CardContent>
  //             <form onSubmit={handleForgotPassword} className="space-y-4">
  //               <div className="space-y-2">
  //                 <Label htmlFor="forgot-email">Email Address</Label>
  //                 <Input
  //                   id="forgot-email"
  //                   type="email"
  //                   value={forgotPasswordEmail}
  //                   onChange={(e) => setForgotPasswordEmail(e.target.value)}
  //                   placeholder="Enter your email address"
  //                   required
  //                 />
  //               </div>
  //               {error && (
  //                 <Alert variant="destructive">
  //                   <AlertDescription>{error}</AlertDescription>
  //                 </Alert>
  //               )}
  //               <div className="space-y-2">
  //                 <Button type="submit" className="w-full" disabled={forgotPasswordLoading}>
  //                   {forgotPasswordLoading ? "Sending..." : "Send Reset Link"}
  //                 </Button>
  //                 <Button type="button" variant="ghost" className="w-full" onClick={() => setShowForgotPassword(false)}>
  //                   Back to Sign In
  //                 </Button>
  //               </div>
  //             </form>
  //           </CardContent>
  //         </Card>
  //       </div>
  //     </div>
  //   )
  // }

  if (showVerificationMessage) {
    // return (
    //   <div className="min-h-screen bg-background flex items-center justify-center p-4">
    //     <div className="w-full max-w-md">
    //       <div className="text-center mb-8">
    //         <Logo size="md" className="mx-auto mb-4" />
    //         <h1 className="text-2xl font-semibold text-foreground">Check Your Email</h1>
    //         <p className="text-muted-foreground">We've sent you a verification link</p>
    //       </div>

    //       <Card>
    //         <CardHeader className="text-center">
    //           <div className="mx-auto mb-4 w-12 h-12 rounded-full bg-primary/10 flex items-center justify-center">
    //             <Mail className="w-6 h-6 text-primary" />
    //           </div>
    //           <CardTitle>Email Verification Required</CardTitle>
    //           <CardDescription>
    //             We've sent a verification link to <strong>{verificationEmail}</strong>
    //           </CardDescription>
    //         </CardHeader>
    //         <CardContent className="space-y-4">
    //           <Alert>
    //             <AlertCircle className="h-4 w-4" />
    //             <AlertDescription>
    //               Please check your email and click the verification link to activate your account. The link will expire
    //               in 24 hours.
    //             </AlertDescription>
    //           </Alert>

    //           {error && (
    //             <Alert
    //               className={
    //                 error.includes("successfully") ? "border-green-200 bg-green-50" : "border-red-200 bg-red-50"
    //               }
    //             >
    //               <AlertDescription className={error.includes("successfully") ? "text-green-800" : "text-red-800"}>
    //                 {error}
    //               </AlertDescription>
    //             </Alert>
    //           )}

    //           <div className="space-y-2">
    //             <Button
    //               onClick={handleResendVerification}
    //               disabled={resendingVerification}
    //               variant="outline"
    //               className="w-full bg-transparent"
    //             >
    //               {resendingVerification ? "Sending..." : "Resend Verification Email"}
    //             </Button>

    //             <Button onClick={() => setShowVerificationMessage(false)} variant="ghost" className="w-full">
    //               Back to Sign In
    //             </Button>
    //           </div>
    //         </CardContent>
    //       </Card>
    //     </div>
    //   </div>
    // )
  }



  return (
    <div className="min-h-screen bg-background flex flex-col items-center justify-center p-6 selection:bg-primary/10">

      {/* Absolute Header Navigation */}
      <div className="absolute top-8 w-full max-w-7xl px-8 flex justify-between items-center">

        <Logo size="sm" className="opacity-100 hover:opacity-100 transition-opacity" />

        <Button
          variant="ghost"
          className="group text-muted-foreground hover:text-foreground transition-all rounded-full px-4 border border-transparent hover:border-border"
        >
          <ArrowLeft className="w-4 h-4 mr-2 transition-transform group-hover:-translate-x-1" />
          Back to Search
        </Button>

      </div>

      <div className="w-full max-w-[400px] space-y-8 animate-in fade-in slide-in-from-bottom-4 duration-700">

        {/* Minimalist Heading */}
        <div className="text-center space-y-2">
          <h1 className="text-4xl font-bold tracking-tight text-foreground">Ads Center</h1>
          <p className="text-muted-foreground text-sm font-medium">
            Manage your campaigns and analytics.
          </p>
        </div>

        {/* Main Auth Container */}
        <div className="bg-card border border-border/60 rounded-[32px] p-8 shadow-2xl shadow-black/20">

          {error && <div className="relative overflow-hidden rounded-2xl border border-[var(--border)] bg-[oklch(from_var(--secondary)_l_c_h_/_0.3)] p-4 backdrop-blur-md animate-in fade-in slide-in-from-top-2 duration-300 mb-3">
            {/* Subtile background glow using your primary color at very low opacity */}
            <div className="absolute -left-4 -top-4 h-12 w-12 bg-[oklch(from_var(--primary)_l_c_h_/_0.05)] blur-2xl" />

            <div className="flex items-start gap-3">
              {/* Status Indicator */}
              <div className="mt-1 flex h-2 w-2 shrink-0 items-center justify-center">
                <div className="h-full w-full rounded-full bg-[var(--primary)] animate-pulse" />
                <div className="absolute h-2 w-2 rounded-full bg-[var(--primary)] opacity-40 animate-ping" />
              </div>

              <div className="space-y-1">
                <p className="text-[10px] font-bold uppercase tracking-[0.2em] text-[var(--muted-foreground)]">
                  Authentication Error
      </p>
                <p className="text-xs leading-relaxed text-[var(--foreground)] opacity-90 font-medium">
                 {error}
      </p>
              </div>
            </div>
          </div>}

          <form className="space-y-6" onSubmit={handleLogin}>

            {/* Username/Email Field */}
            <div className="space-y-2">
              <Label className="text-[10px] font-bold uppercase tracking-[0.2em] text-muted-foreground ml-1">
                Username or Email
              </Label>
              <div className="relative group">
                <div className="absolute left-4 top-1/2 -translate-y-1/2 text-muted-foreground  transition-colors">
                  <User className="w-4 h-4" />
                </div>
                <Input
                  type="text"
                  className="h-12 pl-11 bg-secondary/20 border-border/40 rounded-2xl  transition-all"
                  placeholder="admin@indussearch.com"
                  required
                  onChange={(e) => setLoginData({ ...loginData, username: e.target.value })}
                />
              </div>
            </div>

            {/* Password Field */}
            <div className="space-y-2">
              <div className="flex justify-between items-center ml-1">
                <Label className="text-[10px] font-bold uppercase tracking-[0.2em] text-muted-foreground">
                  Password
                </Label>
              </div>
              <div className="relative group">
                <div className="absolute left-4 top-1/2 -translate-y-1/2 text-muted-foreground  transition-colors">
                  <Lock className="w-4 h-4" />
                </div>
                <Input
                  type={showPassword ? "text" : "password"}
                  className="h-12 pl-11 pr-12 bg-secondary/20 border-border/40 rounded-2xl  transition-all"
                  placeholder="••••••••"
                  required
                  onChange={(e) => setLoginData({ ...loginData, password: e.target.value })}
                />
                <button
                  type="button"
                  onClick={() => setShowPassword(!showPassword)}
                  className="absolute right-0 top-0 h-full px-4 text-muted-foreground transition-colors"
                >
                  {showPassword ? <EyeOff className="h-4 w-4" /> : <Eye className="h-4 w-4" />}
                </button>
              </div>
            </div>

            {/* Action Button */}
            <Button
              type="submit"
              className="w-full h-12 rounded-2xl bg-primary text-primary-foreground font-bold text-sm shadow-xl shadow-primary/5 hover:opacity-90 transition-all active:scale-[0.98] mt-2"
              disabled={loading}
            >
              {loading ? "Verifying..." : "Sign In"}
            </Button>

            {/* Sub-actions Row */}
            <div className="flex items-center justify-between px-1 pt-2">
              <button
                type="button"
                className="text-xs font-semibold text-muted-foreground  transition-colors"
                onClick={() => {/* Nothing happens as requested */ }}
              >
                Create account
              </button>
              <div className="w-1 h-1 rounded-full bg-border" />
              <button
                type="button"
                className="text-xs font-semibold text-muted-foreground  transition-colors"
                onClick={() => setShowForgotPassword(true)}
              >
                Forgot password?
              </button>
            </div>
          </form>
        </div>

        {/* Security Footer */}
        <div className="flex items-center justify-center gap-4 text-muted-foreground/40">
          <div className="h-px w-8 bg-border/50" />
          <ShieldCheck className="w-5 h-5" />
          <div className="h-px w-8 bg-border/50" />
        </div>
      </div>
    </div>

  )
}
